package ru.yandex.tours.clickhouse

import akka.actor.Scheduler
import ru.yandex.tours.util.{Logging, Randoms}
import ru.yandex.tours.util.lang.Futures._

import scala.collection.mutable
import scala.concurrent.duration.FiniteDuration
import scala.concurrent.{ExecutionContext, Future, Promise}
import scala.util.{Failure, Success}

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 21.04.16
 */
class BatchingClickHouseClient(client: ClickHouseClient,
                               scheduler: Scheduler,
                               interval: FiniteDuration,
                               retryInterval: FiniteDuration,
                               maxAttempts: Int)
                              (implicit ec: ExecutionContext) extends ClickHouseClient with Logging {
  override def database: String = client.database

  private case class Batch(name: String, data: Seq[String], future: Future[Unit]) {
    def ++(d: Seq[String]): Batch = Batch(name, data ++ d, future)

    override def toString: String = s"Batch($name, ${data.size} rows)"
  }

  private val map = new mutable.HashMap[String, Batch]()

  override def query(query: String): Future[Iterator[Seq[String]]] = client.query(query)

  private def readable(query: String) = {
    (if (query.length > 40) query.take(40) + "..." else query).replaceAll("\n", " ")
  }

  private def startFlushTask(query: String, promise: Promise[Unit]) = {
    scheduler.scheduleOnce(interval) {
      flush(query, promise)
    }
  }

  private def tryFlush(query: String, batch: Batch, attempt: Int): Future[Unit] = {
    client.update(query, batch.data).andThen {
      case Failure(t) =>
        if (attempt + 1 >= maxAttempts) {
          log.warn(s"Failed to flush $batch in $maxAttempts attempts for query ${readable(query)}. Last exception:", t)
        } else {
          scheduler.scheduleOnce(retryInterval) {
            tryFlush(query, batch, attempt + 1)
          }
        }
      case Success(t) if attempt > 0 =>
        log.info(s"Flushed $batch on ${attempt + 1} attempt for query ${readable(query)}")
    }
  }

  private def flush(query: String, promise: Promise[Unit]) = map.synchronized {
    map.remove(query) match {
      case Some(batch) =>
        val update = tryFlush(query, batch, attempt = 0)
        update.logTiming(s"Flushing $batch for query ${readable(query)}")
        update.andThen {
          case _ => promise.success(())
        }
      case None =>
        log.info(s"Flusher not found batch for query ${readable(query)}")
    }
  }

  override def update(query: String, data: Seq[String]): Future[Unit] = {
    map.synchronized {
      map.get(query) match {
        case Some(batch) =>
          map += query -> (batch ++ data)
          batch.future
        case None =>
          val promise = Promise.apply[Unit]()
          val batch = Batch(Randoms.nextString(8), data, promise.future)
          map += query -> batch
          startFlushTask(query, promise)
          batch.future
      }
    }
  }
}
