package ru.yandex.tours.clickhouse

import akka.util.Timeout
import com.typesafe.config.Config
import ru.yandex.tours.balancer.Balancer
import ru.yandex.tours.util.Logging
import ru.yandex.tours.util.http.AsyncHttpClient
import ru.yandex.tours.util.lang.RichConfig
import ru.yandex.tours.util.parsing.Tabbed
import spray.http.{StatusCodes, Uri}

import scala.concurrent.duration._
import scala.concurrent.{ExecutionContext, Future}
import scala.io.Source
import scala.util.Try

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 30.06.15
 */
class DefaultClickHouseClient(httpClient: AsyncHttpClient,
                              rootDC: String,
                              protocol: String,
                              allHosts: String,
                              port: Int,
                              user: String,
                              password: String,
                              dbName: String,
                              deadTimeout: FiniteDuration
                             )
                             (implicit ec: ExecutionContext, timeout: Timeout) extends ClickHouseClient with Logging {

  override def database: String = dbName

  private val balancing = new Balancer(allHosts, rootDC, deadTimeout)

  def this(httpClient: AsyncHttpClient, config: Config, rootDC: String)(implicit ec: ExecutionContext) = {
    this(
      httpClient,
      rootDC,
      Try(config.getString("protocol")).getOrElse("http"),
      config.getString("host"),
      config.getInt("port"),
      config.getString("username"),
      config.getString("password"),
      Try(config.getString("database")).getOrElse("tours"),
      Try(config.getFiniteDuration("dead_timeout")).getOrElse(FiniteDuration(5, MINUTES))
    )(ec, Timeout(config.getFiniteDuration("timeout")))
  }

  log.info(s"Using clickhouse proto '$protocol', host: ${balancing.hosts}, port: $port, rootDC: $rootDC with user: $user, db_name: $dbName")

  private def uri(host: String) = Uri(s"$protocol://$host:$port/")
  private val baseQuery =
    Uri.Query(
      "user" -> user,
      "password" -> password,
      "database" -> database
    )

  private def parseBody(body: String) = {
    Source.fromString(body).getLines().map {
      case Tabbed(data @_*) => data
    }
  }

  private def update(hosts: Seq[String], queryStr: String, data: Seq[String]): Future[Unit] = {
    if (hosts.isEmpty) {
      sys.error(s"Failed to execute $queryStr: No alive hosts left")
    } else {
      val fullQuery = uri(hosts.head).withQuery(("query" -> queryStr) +: baseQuery)
      httpClient.post(fullQuery, data.mkString("\n").getBytes).map {
        case (StatusCodes.OK, body) =>
        case (code, body) => throw new Exception(s"Invalid status code: $code, body: $body")
      }.recoverWith {
        case exc =>
          log.error(s"Failed to do update at ${hosts.head}, error: $exc")
          balancing.markAsDead(hosts.head)
          update(hosts.tail, queryStr, data)
      }
    }
  }

  def update(query: String, data: Seq[String] = Nil): Future[Unit] = {
    update(balancing.aliveHosts(), query, data)
  }

  private def query(hosts: Seq[String], queryStr: String): Future[Iterator[Seq[String]]] = {
    if (hosts.isEmpty) {
      sys.error(s"Failed to execute $queryStr: No alive hosts left")
    } else {
      val fullQuery = uri(hosts.head).withQuery(("query" -> queryStr) +: baseQuery)
      httpClient.get(fullQuery).map {
        case (StatusCodes.OK, body) => parseBody(body)
        case (code, body) => throw new Exception(s"Invalid status code: $code, body: $body")
      }.recoverWith {
        case exc =>
          log.error(s"Failed to do query at ${hosts.head}, error: $exc")
          balancing.markAsDead(hosts.head)
          query(hosts.tail, queryStr)
      }
    }
  }

  def query(queryStr: String): Future[Iterator[Seq[String]]] = {
    query(balancing.aliveHosts(), queryStr)
  }
}
