package ru.yandex.tours.direction

import java.time.Month

import org.joda.time.LocalDate
import ru.yandex.tours.geo.base.Region
import ru.yandex.tours.geo.base.region.Id
import ru.yandex.tours.model.direction.ThematicInfo
import ru.yandex.tours.model.direction.Thematics.Thematic
import ru.yandex.tours.model.hotels.Partners
import ru.yandex.tours.model.hotels.Partners.Partner
import ru.yandex.tours.model.{Image, Seasonality, UnknownSeasonality}
import ru.yandex.tours.resorts.Ski

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 12.05.15
 */
case class Direction(region: Region,
                     images: Iterable[Image],
                     squareImages: Iterable[Image],
                     isPromoted: Boolean,
                     noVisa: Boolean,
                     thematics: Iterable[ThematicInfo],
                     partners: Seq[Partner],
                     ski: Option[Ski] = None) {
  def mainImage: Image = images.headOption.getOrElse(sys.error("No images in direction: " + region))

  def isCountry = region.isCountry
  def hasCard = ski.nonEmpty  //now we have cards only for ski resorts

  private lazy val thematicsMap = thematics.map(ti => ti.thematic -> ti).toMap

  def relevance(thematic: Thematic) = thematicsMap.get(thematic).fold(0d)(_.relevance)
  def seasonFor(thematic: Thematic) = thematicsMap.get(thematic).fold(UnknownSeasonality: Seasonality)(_.season)
  def highSeasonFor(thematic: Thematic) = thematicsMap.get(thematic).fold(UnknownSeasonality: Seasonality)(_.highSeason)

  def maxRelevance: Double = thematics.map(_.relevance).max

  def relevance(when: LocalDate): Double = relevance(Month.of(when.getMonthOfYear))
  def relevance(month: Month): Double = {
    thematics.map { t =>
      val season = if (t.season.contains(month)) 1d else 0.1d
      val highSeason = if (t.highSeason.contains(month)) 1.2d else 1d
      t.relevance * season * highSeason
    }.max
  }

  def isToursDirection = partners.exists(Partners.isToursPartner)
  def isRoomsDirection = partners.exists(Partners.isRoomsPartner)

  override def hashCode(): Id = region.id

  override def equals(obj: Any): Boolean = obj match {
    case x: Direction =>
      x.region.id == this.region.id
    case other =>
      false
  }
}
