package ru.yandex.tours.direction

import java.io.{ByteArrayInputStream, InputStream}

import org.apache.commons.codec.binary.Base64
import ru.yandex.extdata.common.meta.DataType
import ru.yandex.tours.extdata.{DataDefWithDependencies, DataTypes}
import ru.yandex.tours.geo.base.region.Tree
import ru.yandex.tours.geo.mapping.GeoMappingHolder
import ru.yandex.tours.model.Image
import ru.yandex.tours.model.direction.ThematicInfo
import ru.yandex.tours.model.BaseModel.ProtoImage
import ru.yandex.tours.model.hotels.Partners
import ru.yandex.tours.resorts.SkiResorts
import ru.yandex.tours.util.IO
import ru.yandex.tours.util.parsing.{BooleanValue, IntValue, Tabbed}
import shapeless._

class Directions(directions: Vector[Direction]) {
  val all = directions
  val promoted = directions.filter(_.isPromoted)

  val asMap = directions.map(r => r.region.id -> r).toMap

  def get(id: Int) = asMap.get(id)
}

object Directions extends DataDefWithDependencies[Directions, Tree :: GeoMappingHolder :: SkiResorts :: HNil] {
  override def dataType: DataType = DataTypes.directions
  override def dependsOn: Set[DataType] = Set(DataTypes.regions, DataTypes.cities, DataTypes.countries, DataTypes.skiResorts)

  private def parseImages(encoded: String): List[Image] = {
    var images = List.newBuilder[Image]
    val is = new ByteArrayInputStream(Base64.decodeBase64(encoded))
    var image = ProtoImage.parseDelimitedFrom(is)
    while (image != null) {
      images += Image.fromProto(image)
      image = ProtoImage.parseDelimitedFrom(is)
    }
    images.result()
  }

  override def parse(is: InputStream, dependencies: Tree :: GeoMappingHolder :: SkiResorts :: HNil): Directions = {
    val tree :: geoMapping :: skiResorts :: HNil = dependencies

    val directions = for {
      Tabbed(IntValue(id), images, squareImages, IntValue(priority), BooleanValue(noVisa), thematics) <- IO.readLines(is)
      if geoMapping.isKnownDestination(id)
      region <- tree.region(id)
    } yield {
      Direction(
        region,
        parseImages(images),
        parseImages(squareImages),
        priority > 0,
        noVisa,
        thematics.split(";").filter(_.nonEmpty).map(ThematicInfo.parse),
        Partners.values.filter(p => geoMapping.getPartnerDestination(p, region.id).nonEmpty).toSeq,
        skiResorts.get(id).map(_.ski)
      )
    }

    new Directions(directions.toVector)
      .ensuring(_.all.nonEmpty, "Directions should not be empty")
  }
}