package ru.yandex.tours.direction

import ru.yandex.extdata.common.meta.DataType
import ru.yandex.tours.extdata.{CompositeDataDef, DataTypes}
import ru.yandex.tours.geo.GeoIndex
import ru.yandex.tours.geo.base.Region
import ru.yandex.tours.geo.base.region.Tree
import ru.yandex.tours.geo.mapping.GeoMappingHolder
import ru.yandex.tours.hotels.HotelsIndex
import ru.yandex.tours.model.search.SearchType.SearchType
import shapeless._

//noinspection AccessorLikeMethodIsEmptyParen
/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 16.06.15
 */
class DirectionsStats(countryPriorities: Priorities,
                      resortPriorities: Priorities,
                      tree: Tree,
                      geoMapping: GeoMappingHolder,
                      similarity: DirectionsSimilarity,
                      hotelsIndex: HotelsIndex) {

  private val countryGeoIndex = new GeoIndex(tree.regions.filter(r => geoMapping.isKnownDestCountry(r.id)).toSeq)

  def getPopularCities(): Seq[Region] = {
    val cities = for {
      cityId <- geoMapping.cityGeoIds.toVector
      region <- tree.region(cityId)
    } yield region

    cities
      .sortBy(region => -resortPriorities.getRating(region.id))
  }

  def getPopularCities(geoId: Int): Seq[Region] = {
    val cities = for {
      country <- tree.country(geoId).toVector
      child <- tree.findChildren(country)
      if geoMapping.isKnownDestCity(child.id)
    } yield child

    cities
      .sortBy(region => -resortPriorities.getRating(region.id))
  }

  def getPopularCities(geoId: Int, context: SearchType): Seq[Region] = {
    val cities = for {
      country <- tree.country(geoId).toVector
      child <- tree.findChildren(country)
      if geoMapping.isKnownDestCity(child.id, context)
    } yield child

    cities
      .sortBy(region => -resortPriorities.getRating(region.id, context))
  }

  def getPopularCountries(): Seq[Region] = {
    val countries = for {
      countryId <- geoMapping.countryGeoIds.toVector
      country <- tree.region(countryId)
    } yield country

    countries
      .sortBy(region => -countryPriorities.getRating(region.id))
  }

  def getPopularCountries(geoId: Int): Seq[Region] = {
    val path = tree.pathToRoot(geoId).toSet
    val countries = for {
      countryId <- geoMapping.countryGeoIds.toVector
      country <- tree.region(countryId)
      if !path.contains(country)
    } yield country

    countries
      .sortBy(region => -countryPriorities.getRating(region.id))
  }

  def getPopularCountries(geoId: Int, context: SearchType): Seq[Region] = {
    val path = tree.pathToRoot(geoId).toSet
    val countries = for {
      countryId <- geoMapping.countryGeoIds.toVector
      country <- tree.region(countryId)
      if !path.contains(country)
    } yield country

    countries
      .sortBy(region => -countryPriorities.getRating(region.id, context))
  }

  /**
   * returns up to 4 countries
   * one for each direction
   */
  def getNearCountries(geoId: Int): Seq[Region] = {
    tree.country(geoId) match {
      case Some(country) =>
        val nearCountries = countryGeoIndex.near(country).filter(_ != country)

        var up = Option.empty[Region]
        var down = Option.empty[Region]
        var left = Option.empty[Region]
        var right = Option.empty[Region]

        nearCountries.takeWhile(_ => up.isEmpty || down.isEmpty || left.isEmpty || right.isEmpty).foreach {
          anotherCountry =>
            val dlat = anotherCountry.latitude - country.latitude
            val dlon = anotherCountry.longitude - country.longitude

            val some = Some(anotherCountry)

            if (dlat > dlon.abs) up = up.orElse(some)
            else if (dlat < -dlon.abs) down = down.orElse(some)
            else if (dlon > 0) right = right.orElse(some)
            else left = left.orElse(some)
        }
        List(up, down, left, right).flatten
      case None => List.empty
    }
  }

  def getSimilarCountries(geoId: Int): Seq[Region] = {
    val path = tree.pathToRoot(geoId).toSet
    path.find(r => geoMapping.isKnownDestCountry(r.id)) match {
      case Some(country) =>
        val countries = for {
          countryId <- geoMapping.countryGeoIds.toVector
          if countryId != country.id
          anotherCountry <- tree.region(countryId)
        } yield anotherCountry
        countries
          .sortBy(region => -similarity.getSimilarity(region.id, country.id))
      case None =>
        List.empty
    }
  }

  /** returns priority in range [0.0, 1.0] */
  def getPriority(geoId: Int): Double = {
    val countryPriority = countryPriorities.getRating(geoId)
    if (countryPriority > 0d) countryPriority
    else resortPriorities.getRating(geoId)
  }

  /** returns priority in range [0.0, 1.0] */
  def getPriority(geoId: Int, context: SearchType): Double = {
    val countryPriority = countryPriorities.getRating(geoId, context)
    if (countryPriority > 0d) countryPriority
    else resortPriorities.getRating(geoId, context)
  }
}

object DirectionsStats extends CompositeDataDef[DirectionsStats,
  Priorities :: Priorities :: HotelsIndex :: Tree :: GeoMappingHolder :: DirectionsSimilarity :: HNil] {

  override def dependsOn: Set[DataType] = Set(DataTypes.regions)

  override def from(dependencies: Priorities :: Priorities :: HotelsIndex :: Tree ::
    GeoMappingHolder :: DirectionsSimilarity :: HNil): DirectionsStats = {

    val countryPriorities :: resortPriorities :: hotelsIndex :: tree :: geoMapping :: similarity :: HNil = dependencies
    new DirectionsStats(countryPriorities, resortPriorities, tree, geoMapping, similarity, hotelsIndex)
  }
}