package ru.yandex.tours.direction

import java.io.InputStream

import ru.yandex.extdata.common.meta.DataType
import ru.yandex.tours.extdata.{DataTypes, DataDef}
import ru.yandex.tours.model.search.SearchType
import ru.yandex.tours.model.search.SearchType.SearchType
import ru.yandex.tours.util.parsing.{DoubleValue, Tabbed, IntValue}

case class Priority(id: Int, totalRating: Double, toursRating: Double, roomsRating: Double, suggestRank: Int) {
  def getRating(context: SearchType): Double = context match {
    case SearchType.TOURS => toursRating
    case SearchType.ROOMS => roomsRating
  }
}

class Priorities(map: Map[Int, Priority]) {
  private val maxRating = if (map.isEmpty) 1d else map.values.map(_.totalRating).max

  def getRating(id: Int): Double = map.get(id).map(_.totalRating).getOrElse(0.0d) / maxRating
  def getRating(id: Int, context: SearchType): Double = map.get(id).fold(0.0d)(_.getRating(context)) / maxRating

  def getSuggestRank(id: Int): Int = map.get(id).map(_.suggestRank).getOrElse(0)
  def all: Iterable[Priority] = map.values
}

sealed abstract class PrioritiesDef(val dataType: DataType) extends DataDef[Priorities] {
  def empty: Priorities = new Priorities(Map.empty)

  override def parse(is: InputStream): Priorities = {
    val lines = scala.io.Source.fromInputStream(is).getLines().drop(1)
    val map = lines.map(parseLine).map(x => x.id -> x).toMap
    new Priorities(map)
  }
  
  private def parseLine(line: String): Priority = line match {
    case Tabbed(IntValue(geoId), _, DoubleValue(totalRating),
      DoubleValue(toursRating), DoubleValue(roomsRating), IntValue(suggestRank), _*) =>
      Priority(geoId, totalRating, toursRating, roomsRating, suggestRank)
    case _ =>
      sys.error(s"Cannot parse line [$line]")
  }
}

object ResortPriorities extends PrioritiesDef(DataTypes.resortPriorities)
object HotelPriorities extends PrioritiesDef(DataTypes.hotelPriorities)
object CountryPriorities extends PrioritiesDef(DataTypes.countryPriorities)
