package ru.yandex.tours.events

import java.io.ByteArrayInputStream
import java.util.zip.GZIPInputStream

import com.google.protobuf.Parser
import org.apache.commons.codec.binary.Base64
import org.apache.commons.lang.StringUtils
import org.joda.time.{DateTime, DateTimeZone, LocalDate}
import ru.yandex.tours.model.BaseModel.Currency
import ru.yandex.tours.model.Languages
import ru.yandex.tours.model.search.SearchProducts.{HotelSnippet, Offer}
import ru.yandex.tours.model.search.SearchResults._
import ru.yandex.tours.model.search._
import ru.yandex.tours.util.GZip._
import ru.yandex.tours.util.ProtoIO
import ru.yandex.tours.util.collections.SimpleBitSet
import ru.yandex.tours.util.parsing.IntValue

import scala.collection.JavaConverters._

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 27.02.15
 */
sealed trait SearchEvent {
  def eventTime: DateTime
  def component: String
  def asMap: Map[String, Any]
  def toTSKV: String = {
    val sb = new StringBuilder
    sb.append("component=").append(component)
    sb.append("\ttimestamp=").append(eventTime.withZone(DateTimeZone.UTC))
    for ((key, value) <- asMap) {
      sb ++= "\t"
      sb ++= key
      sb ++= "="
      sb ++= value.toString
    }
    sb.toString
  }
}

object SearchEvent {
  sealed trait WithSearchResult { this: SearchEvent => }

  case class FoundSnippets(eventTime: DateTime, request: HotelSearchRequest, result: HotelSearchResult)
    extends SearchEvent with WithSearchResult {
    override def component: String = SNIPPETS_SEARCH
    override def asMap: Map[String, Any] = {
      request.asMap +
        ("count" -> result.getHotelSnippetCount) +
        ("result" -> Base64.encodeBase64String(compress(result.toByteArray)))
    }
    override def toString: String = s"FoundHotels($eventTime, $request, ${result.getHotelSnippetCount} snippets)"
  }

  case class FoundOffers(eventTime: DateTime, request: OfferSearchRequest, result: OfferSearchResult)
    extends SearchEvent with WithSearchResult {
    override def component: String = OFFERS_SEARCH
    override def asMap: Map[String, Any] = {
      request.asMap +
        ("count" -> result.getOfferCount) +
        ("result" -> Base64.encodeBase64String(compress(result.toByteArray)))
    }
    override def toString: String = s"FoundOffers($eventTime, $request, ${result.getOfferCount} offers)"
  }

  case class FoundFlights(eventTime: DateTime, request: FlightSearchRequest, result: FlightSearchResult)
    extends SearchEvent {
    override def component: String = FLIGHTS_SEARCH
    override def asMap: Map[String, Any] = {
      request.asMap +
        ("count" -> result.getFlightTripsCount) +
        ("result" -> Base64.encodeBase64String(compress(result.toByteArray)))
    }

    override def toString: String = s"FoundFlights($eventTime, $request, ${result.getFlightTripsCount} trips)"
  }

  case class FoundTransfers(eventTime: DateTime, request: TransferSearchRequest, result: TransferSearchResult)
    extends SearchEvent {
    override def component: String = TRANSFER_SEARCH
    override def asMap: Map[String, Any] = {
      request.asMap +
        ("count" -> result.getTransferOptionCount) +
        ("result" -> Base64.encodeBase64String(compress(result.toByteArray)))
    }
  }

  case class Actualized(eventTime: DateTime, request: GetOfferRequest, result: ActualizedOffer) extends SearchEvent {
    override def component: String = TOUR_ACTUALIZATION
    override def asMap: Map[String, Any] =
      request.asMap + ("result" -> Base64.encodeBase64String(compress(result.toByteArray)))
    override def toString: String = s"Actualized($eventTime, $request, actualized: ${result.hasActualizationInfo})"
  }

  case class OstrovokResult(eventTime: DateTime, request: HotelSearchRequest, hotelIds: Seq[String],
                            result: OstrovokSearchResult)
    extends SearchEvent with WithSearchResult {
    override def component: String = if (result.getIsFenced) OSTROVOK_FENCED else OSTROVOK_REGULAR

    override def asMap: Map[String, Any] = {
      request.asMap +
        ("count" -> result.getOffersCount) +
        ("result" -> Base64.encodeBase64String(compress(result.toByteArray))) +
        ("hotel_id" -> hotelIds.mkString(","))
    }
  }

  val SNIPPETS_SEARCH = "hotels_search" // old names
  val OFFERS_SEARCH = "tours_search"    // for backard compatibility
  val FLIGHTS_SEARCH = "flights_search"
  val TRANSFER_SEARCH = "transfer_search"
  val TOUR_ACTUALIZATION = "tour_actualization"
  val OSTROVOK_FENCED = "ostrovok_fenced"
  val OSTROVOK_REGULAR = "ostrovok_regular"

  /** parse event generated by toTSKV method */
  def parseLine(line: String): SearchEvent = {
    val map = line.split("\t").flatMap { part =>
      StringUtils.splitPreserveAllTokens(part, "=", 2) match {
        case Array(key, value) => Some(key -> value)
        case _ => None
      }
    }.toMap
    parseEvent(DateTime.parse(map("timestamp")), map("component"), map)
  }

  /** parse event written by HydraLogger */
  def parseEvent(time: DateTime, component: String, map: Map[String, String]): SearchEvent = {
    component match {
      case SNIPPETS_SEARCH => parseSnippets(time, map)
      case OFFERS_SEARCH => parseOffers(time, map)
      case FLIGHTS_SEARCH => parseFlights(time, map)
      case TRANSFER_SEARCH => parseTransfers(time, map)
      case TOUR_ACTUALIZATION => parseActualization(time, map)
      case OSTROVOK_REGULAR => parseOstrovok(time, map)
      case OSTROVOK_FENCED => parseOstrovok(time, map)
    }
  }

  def parseOstrovok(time: DateTime, map: Map[String, String]) : OstrovokResult = {
    val req = parseSnippetRequest(map)
    val result =
      if (map.contains("result")) parseResult(map, OstrovokSearchResult.PARSER)
      else OstrovokSearchResult.newBuilder().build()

    val hotelIds =
      if (map.contains("hotel_id")) map("hotel_id").split(",").toSeq else Seq.empty
    OstrovokResult(time, req, hotelIds, result)
  }

  /** parse snippets written by HydraLogger */
  def parseSnippets(time: DateTime, map: Map[String, String]): FoundSnippets = {
    val req = parseSnippetRequest(map)
    val result =
      if (map.contains("result")) parseResult(map, HotelSearchResult.PARSER)
      else restoreHotelSearchResult(time, map)

    FoundSnippets(time, req, result)
  }

  /** parse offers written by HydraLogger */
  def parseOffers(time: DateTime, map: Map[String, String]): FoundOffers = {
    val req = parseSnippetRequest(map)
    val hotelId = map("hotel_id").toInt
    val result =
      if (map.contains("result")) parseResult(map, OfferSearchResult.PARSER)
      else restoreOfferSearchResult(time, map)

    val restored =
      if (!result.hasHotelId) result.toBuilder.setHotelId(hotelId).build()
      else result

    FoundOffers(time, OfferSearchRequest(req, hotelId), restored)
  }

  def parseFlights(time: DateTime, map: Map[String, String]): FoundFlights = {
    val req = parseSnippetRequest(map)
    val airportId = map("airport_id")
    val result = parseResult(map, FlightSearchResult.PARSER)

    FoundFlights(time, FlightSearchRequest(req, airportId), result)
  }

  def parseTransfers(time: DateTime, map: Map[String, String]): FoundTransfers = {
    val hotelReq = parseSnippetRequest(map)
    val hotelId = map("hotel_id").toInt
    val airportId = map("airport_id")
    val result = parseResult(map, TransferSearchResult.PARSER)

    FoundTransfers(time, TransferSearchRequest(hotelReq, hotelId, airportId, None), result)
  }

  /** parse actualization written by HydraLogger */
  def parseActualization(time: DateTime, map: Map[String, String]): Actualized = {
    val req = parseSnippetRequest(map)
    val hotelId = map("hotel_id").toInt
    val offerId = map("tour_id")

    val offer = restoreActualizedOffer(time, parseResult(map, ActualizedOffer.PARSER))

    Actualized(
      time,
      GetOfferRequest(OfferSearchRequest(req, hotelId), offerId),
      offer
    )
  }

  private def parseResult[T](map: Map[String, String], parser: Parser[T]): T = {
    val bytes = Base64.decodeBase64(map("result"))
    val is = new GZIPInputStream(new ByteArrayInputStream(bytes))
    parser.parseFrom(is)
  }

  private def parseDelimited[T <: AnyRef](data: String, parser: Parser[T]): Seq[T] = {
    val bytes = Base64.decodeBase64(data)
    val is = new GZIPInputStream(new ByteArrayInputStream(bytes))
    ProtoIO.loadFromStream(is, parser).toVector
  }

  private def restoreProgress(success: Set[Int], failed: Set[Int]) = {
    SearchProgress.newBuilder()
      .setIsFinished(true)
      .setOperatorTotalCount((success ++ failed).size)
      .setOperatorCompleteCount((success ++ failed).size)
      .setOperatorSkippedCount(0)
      .setOperatorFailedCount(failed.size)
      .setOperatorCompleteSet(SimpleBitSet(success ++ failed).packed)
      .setOperatorSkippedSet(SimpleBitSet(Set.empty).packed)
      .setOperatorFailedSet(SimpleBitSet(failed).packed)
      .addAllOBSOLETEFailedOperators(failed.map(Int.box).asJava)
      .build()
  }

  private val resultInfo = ResultInfo.newBuilder().setIsFromLongCache(false).build()

  private def restoreHotelSearchResult(time: DateTime, map: Map[String, String]) = {
    val snippets = parseDelimited(map("snippets"), HotelSnippet.PARSER)
    val failed = parseFailedOperators(map.get("failed"))

    val success = snippets.flatMap(_.getSourceList.asScala).map(_.getOperatorId).toSet

    val progress = restoreProgress(success, failed)

    HotelSearchResult.newBuilder()
      .setCreated(time.getMillis)
      .setUpdated(time.getMillis)
      .setResultInfo(resultInfo)
      .setProgress(progress)
      .addAllHotelSnippet(snippets.asJava)
      .build()
  }

  private def restoreOfferSearchResult(time: DateTime, map: Map[String, String]) = {
    val offers = parseDelimited(map("tours"), Offer.PARSER)
    val failed = parseFailedOperators(map.get("failed"))

    val success = offers.map(_.getSource).map(_.getOperatorId).toSet

    val progress = restoreProgress(success, failed)

    OfferSearchResult.newBuilder()
      .setCreated(time.getMillis)
      .setUpdated(time.getMillis)
      .setResultInfo(resultInfo)
      .setProgress(progress)
      .addAllOffer(offers.asJava)
      .build()
  }

  private def restoreActualizedOffer(time: DateTime, offer: ActualizedOffer): ActualizedOffer = {
    if (offer.hasCreated) offer
    else offer.toBuilder.setCreated(time.getMillis).build
  }

  private def parseFailedOperators(failed: Option[String]): Set[Int] = {
    for {
      bytes <- failed.toIterable
      bitSet = bytes.toLong
      operatorId <- SimpleBitSet.from(bitSet).toSet
    } yield operatorId
  }.toSet

  /** parses HotelSearchRequest */
  private def parseSnippetRequest(map: Map[String, String]): HotelSearchRequest = {
    def optStr(name: String) = map.get(name)
    def str(name: String) = map.getOrElse(name, "")
    def int(name: String) = str(name).toInt

    val ages = str("ages").split(',').map(_.toInt)
    val lostAges = map.collect {
      case (IntValue(age), null) ⇒ age
    }
    val filter: SearchFilter = SearchFilter.parse(str(SearchFilter.name))

    new HotelSearchRequest(
      from = int("from"),
      to = int("to"),
      nights = int("nights"),
      when = LocalDate.parse(str("when")),
      ages = ages ++ lostAges,
      flexWhen = optStr("flex_when").orElse(optStr("when_flex")).get.toBoolean,
      flexNights = optStr("flex_nights").orElse(optStr("nights_flex")).get.toBoolean,
      currency = optStr("currency").map(Currency.valueOf).getOrElse(Currency.RUB),
      lang = optStr("lang").flatMap(Languages.getByName).getOrElse(Languages.ru),
      filter = filter
    )
  }
}
