package ru.yandex.tours.extdata

import java.io.{File, FileInputStream, InputStream}

import ru.yandex.extdata.common.meta.DataType
import ru.yandex.extdata.common.service.ExtDataService
import ru.yandex.tours.util.IO
import shapeless.HList

import scala.reflect.ClassTag

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 12.05.15
 */
abstract class DataDef[T <: AnyRef : ClassTag] {
  def dataType: DataType
  def parse(is: InputStream): T

  protected def onChange(oldValue: Option[T], newValue: T) = ()

  def fromFile(file: File): T = {
    IO.using(new FileInputStream(file))(parse)
  }

  def from(extDataService: ExtDataService): T = {
    IO.using(extDataService.readData(dataType))(parse)
  }

  def from(extDataService: ExtDataService, extDataUpdateChecker: ExtDataUpdateChecker, lazyLoad: Boolean = false): T = {
    Utils.createProxy(
      creator = from(extDataService),
      lazyLoad,
      dataType,
      dependsOn = Set.empty,
      onChange,
      extDataUpdateChecker
    )
  }
}

abstract class CompositeDataDef[T <: AnyRef : ClassTag, D <: HList] {
  def dependsOn: Set[DataType]
  def from(dependencies: D): T

  def from(dependencies: D, extDataUpdateChecker: ExtDataUpdateChecker, lazyLoad: Boolean = false): T = {
    val (proxy, replace) = Utils.createProxy(from(dependencies), lazyLoad)
    extDataUpdateChecker.subscribe(dependsOn.toSeq: _*) {
      replace(from(dependencies))
    }
    proxy
  }
}

abstract class DataDefWithDependencies[T <: AnyRef : ClassTag, D <: HList] {
  def dataType: DataType
  def dependsOn: Set[DataType]

  def parse(is: InputStream, dependencies: D): T

  protected def onChange(oldValue: Option[T], newValue: T) = ()

  def fromFile(file: File, dependencies: D): T = {
    IO.using(new FileInputStream(file))(parse(_, dependencies))
  }

  def from(dependencies: D, extDataService: ExtDataService): T = {
    IO.using(extDataService.readData(dataType))(parse(_, dependencies))
  }

  def from(dependencies: D, extDataService: ExtDataService, extDataUpdateChecker: ExtDataUpdateChecker, lazyLoad: Boolean = false): T = {
    Utils.createProxy(
      from(dependencies, extDataService),
      lazyLoad,
      dataType,
      dependsOn,
      onChange,
      extDataUpdateChecker
    )
  }
}

abstract class VersionAwareDataDef[T <: AnyRef : ClassTag] {
  def dataType: DataType
  def parse(is: InputStream, version: Int): T

  protected def onChange(oldValue: Option[T], newValue: T) = ()

  def fromFile(file: File): T = {
    IO.using(new FileInputStream(file))(parse(_, 1))
  }

  def from(extDataService: ExtDataService, version: Int): T = {
    IO.using(extDataService.readData(dataType))(parse(_, version))
  }

  def from(extDataService: ExtDataService, extDataUpdateChecker: ExtDataUpdateChecker, lazyLoad: Boolean = false): T = {
    Utils.createProxy(
      from(extDataService, extDataUpdateChecker.getVersion(dataType)),
      lazyLoad,
      dataType,
      Set.empty,
      onChange,
      extDataUpdateChecker
    )
  }
}