package ru.yandex.tours.extdata

import java.util.concurrent.TimeUnit

import org.apache.http.client.config.RequestConfig
import org.apache.http.config.SocketConfig
import org.apache.http.impl.client.HttpClientBuilder
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager
import ru.yandex.extdata.common.service.ExtDataService
import ru.yandex.extdata.provider.cache.LocalFSDataCache
import ru.yandex.extdata.provider.{HttpExtDataClient, RemoteExtDataService}
import ru.yandex.tours.app.{AkkaSupport, Application, MetricsSupport}
import ru.yandex.tours.util.file._
import ru.yandex.tours.util.lang.RichConfig

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 20.02.15
 */
trait ExtDataSupport { this: Application =>
  def extDataService: ExtDataService
  def extDataUpdateChecker: ExtDataUpdateChecker
}

trait LocalExtDataSupport extends ExtDataSupport { app: Application =>
  lazy val extDataService: ExtDataService = new LocalExtDataService(app.dataFolder / "data")
  lazy val extDataUpdateChecker: ExtDataUpdateChecker = ExtDataUpdateChecker.empty
}

trait RemoteExtDataSupport extends ExtDataSupport { app: Application with AkkaSupport with MetricsSupport =>
  private lazy val extDataClient = {
    val serviceUrl = app.config.getString("tours.ext.data.http.url")
    val connections = app.config.getInt("tours.ext.data.http.max-connections")
    val timeout = app.config.getDuration("tours.ext.data.http.timeout", TimeUnit.MILLISECONDS)

    val socketConfig: SocketConfig = SocketConfig.custom.setSoTimeout(timeout.toInt).setSoKeepAlive(true).build
    val requestConfig: RequestConfig = RequestConfig.custom.setConnectTimeout(timeout.toInt).build
    val httpClient = HttpClientBuilder.create
      .setMaxConnTotal(connections)
      .setMaxConnPerRoute(connections)
      .setDefaultRequestConfig(requestConfig)
      .setDefaultSocketConfig(socketConfig)
      .build

    new HttpExtDataClient(httpClient, serviceUrl)
  }
  private lazy val localFSCache = {
    val cache = new LocalFSDataCache(
      DataTypes.registry,
      extDataClient,
      app.dataFolder.toString,
      false
    )
    cache.afterPropertiesSet()
    cache
  }

  lazy val extDataService: ExtDataService = new RemoteExtDataService(localFSCache)

  lazy val extDataUpdateChecker: ExtDataUpdateChecker = {
    val checkInterval = app.config.getFiniteDuration("tours.ext.data.check.interval")
    val checker = new RemoteExtDataUpdateChecker(extDataClient, localFSCache, checkInterval)(akkaSystem)
    app.onStop { checker.stop() }
    checker
  }
}