package ru.yandex.tours.extdata

import akka.actor.ActorSystem
import ru.yandex.extdata.common.meta.DataType
import ru.yandex.extdata.provider.HttpExtDataClient
import ru.yandex.extdata.provider.cache.LocalFSDataCache
import ru.yandex.tours.util.{Logging, Monitorings}

import scala.collection.concurrent.TrieMap
import scala.collection.mutable
import scala.concurrent.duration._
import scala.util.control.NonFatal

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 08.03.15
 */
trait ExtDataUpdateChecker {
  def getVersion(dataType: DataType): Int
  def subscribe(dataTypes: DataType*)(action: => Unit): Unit
}

object ExtDataUpdateChecker {
  val empty: ExtDataUpdateChecker = new ExtDataUpdateChecker {
    override def getVersion(dataType: DataType): Int = 1
    override def subscribe(dataTypes: DataType*)(action: => Unit): Unit = ()
  }
}

abstract class AbstractExtDataUpdateChecker extends ExtDataUpdateChecker with Logging {

  protected val subscribers = TrieMap.empty[DataType, Seq[() => Unit]]
  protected val versions = mutable.HashMap.empty[DataType, Int]

  protected def beforeCallbacks(updated: Set[DataType]) = ()
  protected def getLastVersion(dataType: DataType): Int

  override def getVersion(dataType: DataType): Int = versions.getOrElse(dataType, getLastVersion(dataType))

  def updated(dataTypes: Map[DataType, Int]): Unit = {
    beforeCallbacks(dataTypes.keySet)

    val oldVersions = versions.toMap
    try {
      versions ++= dataTypes
      val callbacks = dataTypes.keys.flatMap(dt => subscribers.getOrElse(dt, Seq.empty))
      callbacks.foreach(_.apply())
    } catch {
      case NonFatal(t) =>
        versions ++= oldVersions
        log.warn(s"Failed to execute callbacks for updated data: $dataTypes", t)
    }
  }

  def subscribe(dataTypes: DataType*)(action: => Unit): Unit = {
    val callback = () => action
    for (dt <- dataTypes) {
      val old = subscribers.getOrElse(dt, Seq.empty)
      subscribers.update(dt, old :+ callback)
    }
  }
}

class RemoteExtDataUpdateChecker(extDataClient: HttpExtDataClient,
                                 localFSCache: LocalFSDataCache,
                                 interval: FiniteDuration)
                                (implicit akka: ActorSystem) extends AbstractExtDataUpdateChecker {

  import akka.dispatcher

  override protected def beforeCallbacks(updated: Set[DataType]): Unit =
    Utils.invalidateLocalCache(localFSCache, updated)

  versions ++= Utils.getVersions(localFSCache)
  log.info("Current versions: " + versions)

  private val lastUpdate = Monitorings("data-updater").lastEventWithMarker(
    name = "last-update",
    warningMaxSilence = org.joda.time.Duration.standardMinutes(30),
    errorMaxSilence = org.joda.time.Duration.standardHours(3)
  )

  private val task = akka.scheduler.schedule(interval, interval) {
    try checkNewVersions()
    catch {
      case NonFatal(t) => log.warn("Failed to check new versions", t)
    }
  }

  override protected def getLastVersion(dataType: DataType): Int = {
    val formatVersion = dataType.getCurrentFormatVersion
    val meta = extDataClient.getMetaData(dataType)
    if ((meta ne null) && meta.isDataLoaded(formatVersion)) {
      val instance = meta.getLastDataInstance(formatVersion)
      instance.getVersion
    } else {
      sys.error(s"Data not loaded for $dataType")
    }
  }

  def checkNewVersions(): Unit = {
    var updated = Map.empty[DataType, Int]
    for ((dataType, subs) <- subscribers) {
      val oldVersion = versions.getOrElse(dataType, 0)
      val formatVersion = dataType.getCurrentFormatVersion
      val meta = extDataClient.getMetaData(dataType)

      if ((meta ne null) && meta.isDataLoaded(formatVersion)) {
        val instance = meta.getLastDataInstance(formatVersion)
        val newVersion = instance.getVersion

        if (newVersion > oldVersion) {
          log.info(s"Updated data $dataType. New version: $newVersion")
          updated += dataType -> newVersion
        }
      }
    }
    if (updated.nonEmpty) {
      log.info("Updated versions: " + updated)
      this.updated(updated)
    }
    lastUpdate.mark()
  }

  def stop(): Unit = {
    log.info("Stopping RemoteExtDataUpdateChecker")
    task.cancel()
  }
}
