package ru.yandex.tours.extdata

import java.io.{File, FileOutputStream, InputStream}
import java.util.zip.GZIPInputStream

import org.apache.commons.io.IOUtils
import ru.yandex.extdata.common.meta.DataType
import ru.yandex.tours.util.file._
import ru.yandex.tours.util.{IO, Logging}

import scala.reflect.ClassTag

abstract class FileBaseHolder[T <: AnyRef : ClassTag](dir: File, val dataType: DataType, useGz: Boolean)
  extends VersionAwareDataDef[T] with Logging {

  if (!dir.exists() && !dir.mkdirs()) {
    throw new Exception(s"[${dir.getAbsolutePath}] dir for $dataType doesn't exists")
  }

  override def parse(is: InputStream, version: Int): T = {
    val toRemove = dir.listFiles().filter(_.getName.startsWith(filePrefix)).sortBy(-_.lastModified())
    if (toRemove.length > 1) {
      // Delete all old file except newest one, because it can be at use now.
      toRemove.tail.foreach { f =>
        log.info(s"Deleting old $dataType index: " + f.getAbsolutePath)
        f.delete()
      }
    }

    val fileName = s"${filePrefix}_$version"
    val file = dir / fileName
    if (!file.exists()) {
      val tmpFile = dir / (fileName + ".tmp")

      IO.using(new FileOutputStream(tmpFile)) { os => IOUtils.copy(if (useGz) new GZIPInputStream(is) else is, os) }
      val renamed = tmpFile.renameTo(file)
      if (!renamed) throw new RuntimeException(s"Failed to rename $tmpFile to $fileName")
    }
    parseFile(file)
  }

  protected def parseFile(file: File): T

  private val filePrefix: String = s"${dataType}_${dataType.getCurrentFormatVersion}"
}