package ru.yandex.tours.extdata

import java.io.File
import java.util.concurrent.atomic.AtomicReference
import java.util.{Set => JSet}

import org.springframework.aop.TargetSource
import org.springframework.aop.framework.ProxyFactory
import ru.yandex.extdata.common.meta.DataType
import ru.yandex.extdata.provider.cache.LocalFSDataCache
import ru.yandex.tours.util.Logging
import ru.yandex.tours.util.parsing.IntValue

import scala.collection.JavaConverters._
import scala.reflect._

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 17.03.15
 */
object Utils extends Logging {

  private val updateDataMethod = classOf[LocalFSDataCache].getDeclaredMethod("updateData", classOf[JSet[DataType]])
  private val extDataClientField = classOf[LocalFSDataCache].getDeclaredField("extDataClient")
  private val cacheField = classOf[LocalFSDataCache].getDeclaredField("cache")
  updateDataMethod.setAccessible(true)
  extDataClientField.setAccessible(true)
  cacheField.setAccessible(true)

  def invalidateLocalCache(cache: LocalFSDataCache, dataTypes: Set[DataType]): Unit = {
    if (extDataClientField.get(cache) ne null) {
      val set: JSet[DataType] = dataTypes.asJava
      updateDataMethod.invoke(cache, set)
    }
  }

  def getVersions(cache: LocalFSDataCache): Map[DataType, Int] = {
    if (extDataClientField.get(cache) eq null) return Map.empty
    val FileNameFormat = "(.*)-(.*)-(.*)".r
    val map = cacheField.get(cache).asInstanceOf[java.util.Map[DataType, File]]
    map.asScala.mapValues(_.getName).flatMap {
      case (dataType, FileNameFormat(_, IntValue(format), IntValue(version))) =>
        Some(dataType -> version)
      case (dataType, fileName) =>
        log.warn(s"Unparsable file name [$fileName] for dataType [$dataType]")
        None
    }.toMap
  }

  /** creates proxy and method to replace target */
  def createProxy[T <: AnyRef : ClassTag](target: => T, lazyLoad: Boolean): (T, T => T) = {
    val ts = new HotSwappableLazyTargetSource[T](target, lazyLoad)
    val replacer: (T) => T = ts.swap
    val proxy = ProxyFactory.getProxy(ts).asInstanceOf[T]
    (proxy, replacer)
  }

  def createProxy[T <: AnyRef : ClassTag](creator: => T, lazyLoad: Boolean, dataType: DataType, dependsOn: Set[DataType], onChange: (Option[T], T) => Unit, extDataUpdateChecker: ExtDataUpdateChecker): T = {
    val (proxy, replace) = createProxy(creator, lazyLoad)
    extDataUpdateChecker.subscribe((dependsOn + dataType).toSeq: _*) {
      val created = creator
      val old = replace(created)
      onChange(Option(old), created)
    }
    proxy
  }
  private class HotSwappableLazyTargetSource[T <: AnyRef : ClassTag](creator: => T, lazyLoad: Boolean) extends TargetSource {
    private val target = new AtomicReference[T]()
    override val getTargetClass: Class[_] = classTag[T].runtimeClass
    override def isStatic: Boolean = false

    if (!lazyLoad) target.set(creator)

    def swap(newTarget: T): T = target.getAndSet(newTarget)
    override def getTarget: AnyRef = {
      target.get match {
        case null =>
          target.synchronized {
            if (target.get() eq null) target.set(creator)
            target.get
          }
        case t => t
      }
    }
    override def releaseTarget(target: scala.Any): Unit = {}
  }
}
