package ru.yandex.tours.filter

import ru.yandex.extdata.common.meta.DataType
import ru.yandex.tours.extdata.{CompositeDataDef, DataTypes}
import ru.yandex.tours.filter.State.{AllowedState, Hidden, StringValues}
import ru.yandex.tours.hotels.HotelsIndex
import ru.yandex.tours.model.filter.hotel.StarFilter
import ru.yandex.tours.model.search.SearchProducts.{HotelSnippet, Offer}
import ru.yandex.tours.model.filter.{Filter, FilterableValue, HotelFilter, SnippetFilter}
import ru.yandex.tours.model.search.SearchType.SearchType
import ru.yandex.tours.model.search._
import shapeless._

import scala.collection.mutable

class SnippetFiltrator(hotelsIndex: HotelsIndex, hiddenFilters: HiddenFilters) {
  private val snippetHoldersMap = Filters.snippetHolders.map(h ⇒ h.name → h).toMap
  private val hotelHoldersMap = Filters.hotelHolders.map(h ⇒ h.name → h).toMap
  private val filterHoldersMap = Filters.filterHolders.map(h ⇒ h.name → h).toMap

  def filter(searchRequest: HotelSearchRequest,
             searchType: SearchType,
             snippets: Iterable[HotelSnippet],
             filters: Iterable[Filter]): Iterable[HotelSnippet] = {
    val enabled = hiddenFilters.classify(searchRequest, filters, searchType).enabled
    val (hotelFilters, snippetFilters) = splitFilters(enabled)
    filterHotels(filterSnippets(snippets, snippetFilters), hotelFilters)
  }

  def filterOffers(searchRequest: OfferSearchRequest,
                   searchType: SearchType,
                   offers: Iterable[Offer],
                   filters: Iterable[SnippetFilter]): Iterable[Offer] = {
    offers.filter(offer ⇒ filters.forall(_.fits(offer)))
  }

  def getAllowedState(request: HotelSearchRequest,
                      snippets: Iterable[HotelSnippet],
                      filters: Iterable[Filter],
                      searchType: SearchType): Seq[AllowedState] = {
    val enabledAndVisible = hiddenFilters.classify(request, filters, searchType)
    val (hotelFilters, snippetFilters) = splitFilters(enabledAndVisible.enabled)
    val filteredBySnippets = filterSnippets(snippets, snippetFilters)
    val filteredByHotels = filterHotels(snippets, hotelFilters)
    val states = Seq(
      getSnippetFiltersAllowedState(filteredByHotels, snippetFilters),
      getHotelFiltersAllowedState(filteredBySnippets, hotelFilters),
      enabledAndVisible.hidden.map(Hidden)
    ).flatten
    // HOTELS-2397: hack for stars filter state
    if (SearchFilter(filters, searchType) != EmptySearchFilter)
      states.map(starFilterStateHack)
    else states
  }

  private val allStarFilterValues = 1 to 5 map (_.toString)
  private def starFilterStateHack(state: AllowedState): AllowedState = {
    state match {
      case s@StringValues(StarFilter.name, _) => s.copy(values = allStarFilterValues)
      case s => s
    }
  }

  private def splitFilters(filters: Iterable[Filter]): (Iterable[HotelFilter], Iterable[SnippetFilter]) = {
    val hotelFilters = filters.collect { case x: HotelFilter ⇒ x }
    val snippetFilters = filters.collect { case x: SnippetFilter ⇒ x }
    (hotelFilters, snippetFilters)
  }

  private def getSnippetFiltersAllowedState(snippets: Iterable[HotelSnippet],
                                            filters: Iterable[SnippetFilter]): Seq[AllowedState] = {
    getAllowedState[HotelSnippet, SnippetFilter](snippets, filters, {
      case (snippet, filter) ⇒ filter.fits(snippet)
    }, {
      case (snippet, filter) ⇒ snippetHoldersMap(filter.name).getValues(snippet)
    })
  }

  private def getHotelFiltersAllowedState(snippets: Iterable[HotelSnippet],
                                          filters: Iterable[HotelFilter]): Seq[AllowedState] = {
    val hotelsIds = snippets.map(_.getHotelId)
    val result = for {
      hotelFilter ← filters
      holder ← hotelHoldersMap.get(hotelFilter.name)
    } yield {
      val otherFilters = filters.filter(_ != hotelFilter)
      val values = holder.getValues.filter { value ⇒
        val checkedFilter = holder.construct(Seq(value))
        hotelsIndex.getHotelsCount(hotelsIds, otherFilters ++ Iterable(checkedFilter), None) > 0
      }
      holder.toState(values.toSet)
    }
    result.toSeq
  }

  def getAllowedStateForOffers(offers: Iterable[Offer],
                               filters: Iterable[SnippetFilter]): Seq[AllowedState] = {
    getAllowedState[Offer, SnippetFilter](offers, filters, {
      case (offer, filter) => filter.fits(offer)
    }, {
      case (offer, filter) => snippetHoldersMap(filter.name).getValues(offer)
    })
  }

  private def filterSnippets(snippets: Iterable[HotelSnippet],
                             filters: Iterable[SnippetFilter]): Iterable[HotelSnippet] = {
    snippets.flatMap { snippet ⇒
      filters.foldLeft(Option(snippet)) {
        case (optSnippet, filter) ⇒ optSnippet.flatMap(filter.amendIfFits)
      }.toSeq
    }
  }

  private def filterHotels(snippets: Iterable[HotelSnippet],
                           filters: Iterable[HotelFilter]): Iterable[HotelSnippet] = {
    val hotelIds = snippets.map(_.getHotelId).toSet
    val filtered = hotelsIndex.filter(hotelIds, filters, None)
    snippets.filter(snippet ⇒ filtered.contains(snippet.getHotelId))
  }

  private def getAllowedState[O, F <: Filter](objects: Iterable[O],
                                              filters: Iterable[F],
                                              fits: (O, F) ⇒ Boolean,
                                              values: (O, F) ⇒ Seq[FilterableValue]) = {

    val filter2values = mutable.Map.empty[F, mutable.Set[FilterableValue]]
    filters.foreach(f ⇒ filter2values.put(f, mutable.Set.empty))

    val filterArray = filters.toIndexedSeq
    objects.foreach { obj ⇒
      val fitArray = filterArray.map(f ⇒ fits(obj, f))
      val matched = fitArray.count(_ == true)
      filterArray.zipWithIndex.foreach { case (filter, index) ⇒
        val dec = if (fitArray(index)) 0 else 1
        // That means, that all filters except current are matched
        if (matched + dec == fitArray.length) {
          filter2values(filter) ++= values(obj, filter)
        }
      }
    }
    filter2values.map {
      case (filter, vals) ⇒ filterHoldersMap(filter.name).toState(vals.toSet)
    }.toSeq
  }
}

object SnippetFiltrator extends CompositeDataDef[SnippetFiltrator, HotelsIndex :: HiddenFilters :: HNil] {
  override def dependsOn: Set[DataType] = DataTypes.shardedHotels.dataTypes.toSet ++ Set(DataTypes.hiddenFilters)

  override def from(dependencies: HotelsIndex :: HiddenFilters :: HNil): SnippetFiltrator = {
    val hotelsIndex :: hiddenFilters :: HNil = dependencies
    new SnippetFiltrator(hotelsIndex, hiddenFilters)
  }
}
