package ru.yandex.tours.geo

import java.io.InputStream

import ru.yandex.extdata.common.meta.DataType
import ru.yandex.tours.extdata.{DataDefWithDependencies, DataTypes}
import ru.yandex.tours.geo.base.region
import ru.yandex.tours.geo.base.region.{Tree, Types}
import ru.yandex.tours.geo.mapping.GeoMappingHolder
import ru.yandex.tours.search.DefaultRequestGenerator
import ru.yandex.tours.util.Collections._
import ru.yandex.tours.util.collections.Bag
import ru.yandex.tours.util.lang.SideEffectOption
import ru.yandex.tours.util.parsing.{IntValue, Tabbed}
import ru.yandex.tours.util.Logging
import shapeless._

import scala.collection.mutable
import scala.collection.mutable.ListBuffer

/* @author berkut@yandex-team.ru */
class Departures(departures: Map[Int, Seq[Int]],
                 departuresToCountry: Map[(Int, Int), Seq[Int]],
                 tree: region.Tree,
                 knownIds: Set[Int]) extends Logging {

  private val country2airports = {
    for {
      id <- knownIds
      region <- tree.region(id) onEmpty { onUnknownAirport(id) }
      parent <- tree.country(region) onEmpty { onUnknownAirport(id) }
    } yield parent -> region
  }.toMultiMap

  def getNearestAirport(geoId: Int): Int = {
    if (knownIds.contains(geoId)) return geoId
    val result = for {
      region <- tree.region(geoId)
      country <- tree.country(region)
      airports <- country2airports.get(country)
    } yield airports.minBy(airport => ru.yandex.tours.geo.distanceInKm(airport, region)).id
    result.getOrElse(DefaultRequestGenerator.MOSCOW_GEO_ID)
  }

  def getDepartures(geoId: Int): List[Int] = {
    if (geoId <= 0) return List(geoId)
    val result = ListBuffer.empty[Int]
    if (knownIds.contains(geoId)) result += geoId
    result ++= departures.getOrElse(geoId, Nil)
    if (result.isEmpty) result += getNearestAirport(geoId)
    result += DefaultRequestGenerator.MOSCOW_GEO_ID
    result.result().distinct
  }

  def getDepartures(geoId: Int, toRegion: Int): List[Int] = {
    if (geoId <= 0) return List(geoId)
    val result = ListBuffer.empty[Int]
    for {
      toCountry <- tree.country(toRegion)
      departures <- departuresToCountry.get((geoId, toCountry.id))
    } result ++= departures
    result ++= getDepartures(geoId)
    result += DefaultRequestGenerator.MOSCOW_GEO_ID
    result.result().distinct
  }

  def getBestDeparture(geoId: Int): Int = {
    getDepartures(geoId).head
  }

  def getBestDeparture(geoId: Int, toRegion: Int): Int = {
    getDepartures(geoId, toRegion).head
  }

  protected def onUnknownAirport(id: Int) {
    log.error(s"Unknown geoId or country for known departure cities: $id")
  }
}

object Departures extends DataDefWithDependencies[Departures, Tree :: GeoMappingHolder :: HNil] with Logging {
  override def dataType: DataType = DataTypes.departureStats
  override def dependsOn: Set[DataType] = Set(DataTypes.regions, DataTypes.departures)

  override def parse(is: InputStream, dependencies: Tree :: GeoMappingHolder :: HNil): Departures = {
    val tree :: geoMapping :: HNil = dependencies
    val departures = new mutable.HashMap[Int, Bag[Int]]
    val departuresToCountry = new mutable.HashMap[(Int, Int), Bag[Int]]

    scala.io.Source.fromInputStream(is).getLines().foreach {
      case Tabbed(IntValue(rid), IntValue(from), IntValue(to), IntValue(count)) =>
        departures.getOrElseUpdate(rid, new Bag) += (from, count)
        departuresToCountry.getOrElseUpdate((rid, to), new Bag) += (from, count)
    }

    def relevant(bag: Bag[Int]) = {
      val sum = bag.sum
      bag.filterByCount(c => c > 3 && c.toDouble / sum > 0.1).keysDesc
    }

    new Departures(
      departures.mapValues(relevant).toMap,
      departuresToCountry.mapValues(relevant).toMap,
      tree,
      geoMapping.departuresGeoIds
    )
  }
}
