package ru.yandex.tours.geo

import ru.yandex.tours.model.MapRectangle
import ru.yandex.tours.model.geo.MapObject

class GeoIndex[T <: MapObject](objects: Iterable[T]) {
  private val index = objects.groupBy(t => (t.longitude.toInt, t.latitude.toInt))

  def inRectangle(mapInfo: MapRectangle): Seq[T] = {
    val candidates = mapInfo.coordinateIndexes.flatMap({case (lon, lat) => index.getOrElse((lon, lat), List.empty)})
    candidates.filter(obj => mapInfo.contains(obj.longitude, obj.latitude))
  }

  def near(obj: MapObject): Iterator[T] = {
    GeoIndex.coordinateIndexesDiff.iterator.flatMap {
      case (dlon, dlat) =>
        val lon = ((obj.longitude.toInt + dlon) + 180) % 360 - 180
        val lat = obj.latitude.toInt + dlat
        if (lat.abs > 90) Nil
        else index.getOrElse((lon, lat), List.empty).toSeq.sortBy(distanceInKm(_, obj))
    }
  }
}

object GeoIndex {
  lazy val coordinateIndexesDiff = {
    val indexes = for {
      lon <- -10 to 10
      lat <- -10 to 10
    } yield (lon, lat)
    indexes.sortBy { case (lon, lat) => distanceInKm(lat, lon, 0, 0) }
  }
}