package ru.yandex.tours.geo.base

import org.joda.time.DateTimeZone
import ru.yandex.tours.geo.base.region.{Id, Types}
import ru.yandex.tours.model.{Languages, LocalizedString, MapRectangle}
import ru.yandex.tours.util.parsing.Tabbed

/* @author berkut@yandex-team.ru */

/** Geobase region.
  */
case class Region(id: region.Id,
                  `type`: region.Type,
                  parentId: region.Id,
                  name: LocalizedString,
                  genitive: String, //родительный падеж
                  dative: String, //дательный падеж
                  accusative: String, //винительный падеж
                  locative: String, //предложный падеж
                  preposition: String, //приставка
                  synonyms: Set[String],
                  position: Int, // популярность/важность региона с точки зрения Яндекса. Больше – лучше
                  longitude: Double,
                  latitude: Double,
                  boundingBox: MapRectangle,
                  timeZone: Option[DateTimeZone] = None,
                  childrenIds: Seq[Int] = Seq.empty,
                  isoCode: Option[String] = None) extends RegionBase[region.Id] {

  require(id > 0, s"Non-positive region ID ($id)")
  require(`type` != null, "Null region type")
  require(parentId >= 0, s"Invalid parent region id ($parentId)")
  require(name != null && name.nonEmpty, "Empty region name")
  require(childrenIds.isEmpty || `type` == Types.MetaRegion, "only meta regions should have explicit children")

  def isCountry: Boolean = `type` == Types.Country || `type` == Types.OverseasLand

  private def clean(x: String) = x.replaceAll("\t", " ")

  def toTsv: String = Tabbed(
    id,
    parentId,
    `type`,
    clean(name(Languages.ru)),
    allNames.map(clean).mkString("#"),
    latitude,
    longitude,
    position
  )

  override def hashCode(): Id = id

  override def equals(obj: Any): Boolean = obj match {
    case x: Region =>
      x.id == this.id
    case other =>
      false
  }
}

