package ru.yandex.tours.geo.base.custom

import org.joda.time.DateTimeZone
import ru.yandex.tours.geo.base.Region
import ru.yandex.tours.geo.base.region.Types
import ru.yandex.tours.model.{MapRectangle, LocalizedString}

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 23.11.15
 */
case class PartialRegion(`type`: Option[Types.Value] = None,
                         parentId: Option[Int] = None,
                         name: Option[LocalizedString] = None,
                         genitive: Option[String] = None,
                         dative: Option[String] = None,
                         accusative: Option[String] = None,
                         locative: Option[String] = None,
                         preposition: Option[String] = None,
                         synonyms: Set[String] = Set.empty,
                         longitude: Option[Double] = None,
                         latitude: Option[Double] = None,
                         boundingBox: Option[MapRectangle] = None,
                         timeZone: Option[DateTimeZone] = None) {

  def isComplete = `type`.nonEmpty && parentId.nonEmpty && name.nonEmpty &&
    genitive.nonEmpty && dative.nonEmpty && accusative.nonEmpty && locative.nonEmpty &&
    preposition.nonEmpty && longitude.nonEmpty && latitude.nonEmpty && boundingBox.nonEmpty

  def patch(region: Region) = {
    region.copy(
      `type` = this.`type`.getOrElse(region.`type`),
      parentId = this.parentId.getOrElse(region.parentId),
      name = this.name.getOrElse(region.name),
      genitive = this.genitive.getOrElse(region.genitive),
      dative = this.dative.getOrElse(region.dative),
      accusative = this.accusative.getOrElse(region.accusative),
      locative = this.locative.getOrElse(region.locative),
      preposition = this.preposition.getOrElse(region.preposition),
      synonyms = if (this.synonyms.nonEmpty) this.synonyms else region.synonyms,
      longitude = this.longitude.getOrElse(region.longitude),
      latitude = this.latitude.getOrElse(region.latitude),
      boundingBox = this.boundingBox.getOrElse(region.boundingBox),
      timeZone = this.timeZone.orElse(region.timeZone)
    )
  }

  def toRegion(id: Int, childrenIds: Seq[Int]): Region = {
    require(isComplete, "Region must be fully filled")
    new Region(id, `type`.get, parentId.get, name.get,
      genitive.get, dative.get, accusative.get, locative.get,
      preposition.get, synonyms, 0, longitude.get, latitude.get, boundingBox.get, timeZone,
      childrenIds
    )
  }
}

object PartialRegion {
  import play.api.libs.json._
  import ru.yandex.tours.serialize.json.CommonJsonFormats._

  implicit val jsonFormat = Json.format[PartialRegion]
}