package ru.yandex.tours.geo.base.export

import org.joda.time.DateTimeZone
import org.xml.sax.Attributes
import org.xml.sax.helpers.DefaultHandler
import ru.yandex.tours.geo.base.region.RegionBoundaries
import ru.yandex.tours.geo.base.{Region, region}
import ru.yandex.tours.model.{Languages, LocalizedString, MapRectangle}
import ru.yandex.tours.util.Logging
import ru.yandex.tours.util.parsing.IntValue

import scala.util.Try

/** Sax handler of regions from geobase XML export.
  */
class RegionHandler(regionBoundaries: RegionBoundaries) extends DefaultHandler with Logging {
  private val regions = Vector.newBuilder[Region]

  override def startElement(uri: String, localName: String, qName: String, attrs: Attributes) {
    def optAttr(names: String*) = names.flatMap(n => Option(attrs.getValue(n))).headOption.filter(_.nonEmpty)
    def optIntAttr(name: String) = Option(attrs.getValue(name)).map(_.toInt)
    def optDoubleAttr(name: String) = Option(attrs.getValue(name)).map(_.toDouble)
    def booleanAttr(name: String, default: Boolean = false) = Option(attrs.getValue(name)).map(_.toBoolean).getOrElse(default)
    def attr(names: String*) = optAttr(names: _*).getOrElse("")

    def getBoundingBox(id: Int, lat: Double, lon: Double) = {
      def byBoundaries = for {
        minLat <- optDoubleAttr("min_lat")
        minLon <- optDoubleAttr("min_lon")
        maxLat <- optDoubleAttr("max_lat")
        maxLon <- optDoubleAttr("max_lon")
      } yield MapRectangle(minLon, minLat, maxLon, maxLat)
      def bySpan = for {
        latSpan <- optDoubleAttr("spn_lat")
        lonSpan <- optDoubleAttr("spn_lon")
        box <- Try(MapRectangle.byCenterAndSpan(lon, lat, lonSpan, latSpan)).toOption
      } yield box
      regionBoundaries.getBoundaries(id).orElse(byBoundaries).orElse(bySpan).getOrElse {
        log.warn(s"Invalid map rectangle of region with id $id!")
        MapRectangle.empty
      }
    }

    if ("region".equalsIgnoreCase(qName)) {
      try {
        val optRegion = for {
          id <- optIntAttr("id")
          parentId <- optIntAttr("parent")
          typ <- optIntAttr("type").map(region.Types.apply)
          position <- optIntAttr("position")
          lat <- optDoubleAttr("lat")
          lon <- optDoubleAttr("lon")
        } yield {
            val tz = optAttr("tzname", "tz_name").flatMap(tz => Try(DateTimeZone.forID(tz)).toOption)
            val isoCode = optAttr("short_ename")
            val synonyms = attr("syn").split(",").map(_.trim).filterNot(_.isEmpty).toSet
            val children = attr("children").split(",").map(_.trim).filterNot(_.isEmpty).flatMap(IntValue.apply)
            val genitive = attr("genitive")
            val dative = attr("dative")
            val locative = attr("locative")
            val accusative = attr("accusative")
            val preposition = attr("preposition").intern()
            val names = Map(
              Languages.ru -> attr("name"),
              Languages.tr -> attr("trname", "tr_name"),
              Languages.en -> attr("en_name"),
              Languages.kz -> attr("kz_name"),
              Languages.ua -> attr("ua_name"),
              Languages.by -> attr("byname", "by_name")
            ).filter(_._2.nonEmpty)
            val rectangle = getBoundingBox(id, lat, lon)
            Region(id, typ, parentId, LocalizedString(names), genitive, dative, accusative, locative, preposition,
              synonyms, position, lon, lat, rectangle, tz, children, isoCode)
          }

        for (region <- optRegion)
          regions += region
      }
      catch {
        case e: Exception =>
          log.error("Error while parse region node", e)
      }
    }
  }

  def getRegions: Iterable[Region] = regions.result()
}