package ru.yandex.tours.geo.base.export

import java.io.{File, FileInputStream, InputStream}
import javax.xml.parsers.SAXParserFactory

import ru.yandex.tours.geo.base.Region
import ru.yandex.tours.geo.base.region.{RegionBoundaries, Tree, Types}
import ru.yandex.tours.model.MapRectangle
import ru.yandex.tours.util.Logging

import scala.collection.mutable
import scala.util.Try

/** Parses geobase export to regions tree.
  * Export obtained from '''http://geobase.yandex.ru/geobase.xml?fields=id,name,type,parent,position,syn,ua_name,byname,kz_name,en_name,trname,lat,lon,spn_lat,spn_lon,tzname,genitive,accusative,preposition'''
  */
object XmlParser extends Logging {

  private val saxParserFactory = SAXParserFactory.newInstance

  def parse(file: File, regionBoundaries: RegionBoundaries): Try[Tree] =
    parse(new FileInputStream(file), regionBoundaries)

  def parse(is: InputStream, regionBoundaries: RegionBoundaries): Try[Tree] =
    Try(new Tree(getRegions(is, regionBoundaries)))

  def getRegions(is: InputStream, regionBoundaries: RegionBoundaries): Iterable[Region] = {
    val handler = new RegionHandler(regionBoundaries)
    saxParserFactory.newSAXParser().parse(is, handler)
    fixAndValidate(handler.getRegions)
  }

  private def fixAndValidate(regions: Iterable[Region]) = {
    val errors = mutable.Buffer.empty[String]
    val result = regions
      .withFilter(r ⇒ r.`type` != Types.Other || r.id == 10000 || r.id >= 1000000000)
      .map { region =>
      if (region.latitude > 90 || region.latitude < -90 || region.longitude > 180 || region.longitude < -180) {
        errors += s"Invalid coordinates of region ${region.id}: latitude: ${region.latitude}, longitude: ${region.longitude}"
        region.copy(longitude = 0, latitude = 0, boundingBox = MapRectangle.empty)
      } else {
        region
      }
    }
    if (errors.nonEmpty) {
      log.warn(s"There ${errors.size} problem regions:\n ${errors.mkString("\n")}")
    }
    result
  }
}