package ru.yandex.tours.geo.mapping

import java.io.InputStream

import ru.yandex.extdata.common.meta.DataType
import ru.yandex.tours.extdata.{DataDef, DataTypes}
import ru.yandex.tours.model.hotels.Partners
import ru.yandex.tours.model.hotels.Partners.Partner
import ru.yandex.tours.util.{IO, Logging}
import ru.yandex.tours.util.collections.Bag
import ru.yandex.tours.util.lang._
import ru.yandex.tours.util.parsing.{CSV, IntValue, Tabbed}

import scala.io.Source

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 13.05.15
 * Map is: (partner, id) -> partner_region_id
 */
class GeoMapping(mapping: Map[(Partner, Int), String]) {

  private lazy val reverseMapping: Map[(Partner, String), Int] = mapping.flatMap {
    case ((partner, geoId), CSV(codes @_*)) =>
      codes.map(code => (partner, code) -> geoId)
  }
  val knownIds: Set[Int] = mapping.keys.map(_._2).toSet

  def isKnown(geoId: Int): Boolean = knownIds.contains(geoId)

  def getPartnerRegion(partner: Partner, geoId: Int): Option[String] = mapping.get((partner, geoId))

  def getYaRegion(partner: Partner, geoCode: String): Option[Int] = reverseMapping.get((partner, geoCode))
}

class GeoMappingDef(val dataType: DataType) extends DataDef[GeoMapping] with Logging {
  def apply(map: Map[(Int, Int), String]): GeoMapping = {
    new GeoMapping(map.map(kv => (Partners(kv._1._1), kv._1._2) -> kv._2))
  }

  def empty: GeoMapping = new GeoMapping(Map.empty)

  override def parse(is: InputStream): GeoMapping = {
    val unknownPartners = new Bag[Int]()
    val mapping =
      IO.readLines(is).flatMap {
        case Tabbed(IntValue(partnerId), IntValue(k), v) =>
          Partners.getOpt(partnerId)
            .onEmpty(unknownPartners += partnerId)
            .map(partner => (partner, k) -> v)
      }.toMap
    if (unknownPartners.nonEmpty) {
      log.warn(s"Some geo mappings are ignored due to unknown partnerId : $unknownPartners")
    }
    new GeoMapping(mapping)
  }
}

object CityMapping extends GeoMappingDef(DataTypes.cities)
object CountryMapping extends GeoMappingDef(DataTypes.countries)
object DepartureMapping extends GeoMappingDef(DataTypes.departures)
object OldLtCityMapping extends GeoMappingDef(DataTypes.oldLtCities)
object OldLtCountryMapping extends GeoMappingDef(DataTypes.oldLtCountries)
object OldLtDepartureMapping extends GeoMappingDef(DataTypes.oldLtDepartures)
object AirportMapping extends GeoMappingDef(DataTypes.airports)