package ru.yandex.tours.geo.mapping

import java.io.File

import ru.yandex.extdata.common.service.ExtDataService
import ru.yandex.tours.extdata.ExtDataUpdateChecker
import ru.yandex.tours.model.hotels.Partners
import ru.yandex.tours.model.hotels.Partners.Partner
import ru.yandex.tours.model.search.SearchType.SearchType

/* @author berkut@yandex-team.ru */

object GeoMappingHolder {

  def apply(countries: Map[(Int, Int), String],
            cities: Map[(Int, Int), String],
            departures: Map[(Int, Int), String],
            airports: Map[(Int, Int), String]): GeoMappingHolder = {
    new GeoMappingHolder(
      CountryMapping(countries),
      CityMapping(cities),
      DepartureMapping(departures),
      AirportMapping(airports)
    )
  }

  def from(extDataService: ExtDataService,
           extDataUpdateChecker: ExtDataUpdateChecker,
           lazyLoad: Boolean = false): GeoMappingHolder = {
    new GeoMappingHolder(
      CountryMapping.from(extDataService, extDataUpdateChecker, lazyLoad),
      CityMapping.from(extDataService, extDataUpdateChecker, lazyLoad),
      DepartureMapping.from(extDataService, extDataUpdateChecker, lazyLoad),
      AirportMapping.from(extDataService, extDataUpdateChecker, lazyLoad)
    )
  }

  def from(extDataService: ExtDataService): GeoMappingHolder = {
    new GeoMappingHolder(
      CountryMapping.from(extDataService),
      CityMapping.from(extDataService),
      DepartureMapping.from(extDataService),
      AirportMapping.from(extDataService)
    )
  }

  def oldLtFrom(extDataService: ExtDataService,
                extDataUpdateChecker: ExtDataUpdateChecker,
                lazyLoad: Boolean = false): GeoMappingHolder = {
    new GeoMappingHolder(
      OldLtCountryMapping.from(extDataService, extDataUpdateChecker, lazyLoad),
      OldLtCityMapping.from(extDataService, extDataUpdateChecker, lazyLoad),
      OldLtDepartureMapping.from(extDataService, extDataUpdateChecker, lazyLoad),
      AirportMapping.from(extDataService, extDataUpdateChecker, lazyLoad)
    )
  }

  def parse(cityFile: File, countryFile: File, departureFile: File, airportFile: File): GeoMappingHolder = {
    new GeoMappingHolder(
      CountryMapping.fromFile(countryFile),
      CityMapping.fromFile(cityFile),
      DepartureMapping.fromFile(departureFile),
      AirportMapping.fromFile(airportFile)
    )
  }

  def empty: GeoMappingHolder = {
    new GeoMappingHolder(CountryMapping.empty, CityMapping.empty, DepartureMapping.empty, AirportMapping.empty)
  }
}

class GeoMappingHolder(countryMapping: GeoMapping,
                       cityMapping: GeoMapping,
                       departureCityMapping: GeoMapping,
                       airportMapping: GeoMapping) {

  def countryGeoIds: Set[Int] = countryMapping.knownIds
  def cityGeoIds: Set[Int] = cityMapping.knownIds
  def departuresGeoIds: Set[Int] = departureCityMapping.knownIds

  def getPartnerCountry(partner: Partner, geoId: Int): Option[String] = countryMapping.getPartnerRegion(partner, geoId)
  def getPartnerCity(partner: Partner, geoId: Int): Option[String] = cityMapping.getPartnerRegion(partner, geoId)
  def getPartnerDestination(partner: Partner, geoId: Int): Option[String] = getPartnerCity(partner, geoId).orElse(getPartnerCountry(partner, geoId))
  def getPartnerDeparture(partner: Partner, geoId: Int): Option[String] = departureCityMapping.getPartnerRegion(partner, geoId)
  def getPartnerAirport(partner: Partner, geoId: Int): Option[String] = airportMapping.getPartnerRegion(partner, geoId)

  def getGeoIdByCity(partner: Partner, id: String): Option[Int] = cityMapping.getYaRegion(partner, id).orElse(departureCityMapping.getYaRegion(partner, id))
  def getGeoIdByCountry(partner: Partner, id: String): Option[Int] = countryMapping.getYaRegion(partner, id)
  def getGeoId(partner: Partner, id: String): Option[Int] = getGeoIdByCity(partner, id).orElse(getGeoIdByCountry(partner, id))
  def getAirport(partner: Partner, id: String): Option[Int] = airportMapping.getYaRegion(partner, id)

  def isKnownDestCountry(id: Int): Boolean = countryMapping.isKnown(id)
  def isKnownDestCountry(id: Int, searchType: SearchType): Boolean = {
    Partners.partnersByType(searchType).exists(partner => getPartnerCountry(partner, id).isDefined)
  }

  def isKnownDestCity(id: Int): Boolean = cityMapping.isKnown(id)
  def isKnownDestCity(id: Int, searchType: SearchType): Boolean = {
    Partners.partnersByType(searchType).exists(partner => getPartnerCity(partner, id).isDefined)
  }

  def isKnownDestination(id: Int): Boolean = isKnownDestCountry(id) || isKnownDestCity(id)
  def isKnownDestination(id: Int, searchType: SearchType): Boolean = {
    isKnownDestCountry(id, searchType) || isKnownDestCity(id, searchType)
  }

  def isDepartureCity(id: Int): Boolean = departureCityMapping.isKnown(id)
}
