package ru.yandex.tours.geo.matching

import com.typesafe.config.Config
import ru.yandex.tours.geo.base.region
import ru.yandex.tours.geo.partners.PartnerTree
import ru.yandex.tours.hotels.HotelsIndex

import scala.collection.mutable

/**
  * Created by asoboll on 29.01.16.
  */
class GeoMatcher(config: Config,
                 geobaseTree: region.Tree,
                 hotelsIndex: HotelsIndex,
                 geoSynonyms: Map[Int, Iterable[String]] = Map.empty) {
  private val hotelMatcher = new RegionMatcherByHotelsIndex(config.getConfig("hotels-index-matcher"), geobaseTree, hotelsIndex)
  private val nameMatcher = new NameMatcher(config.getConfig("name-matcher"), geobaseTree, geoSynonyms)
  private val typeFilter = new TypeFilter(config.getConfig("type-filter"))
  private val popularityAmplifier = new PopularityAmplifier(config.getConfig("popularity-amplifier"))
  private val coordsFilter = new CoordsFilter(config.getConfig("coords-filter"))
  private val countryFilter = new CountryFilter(config.getConfig("country-filter"), geobaseTree)
  private val parentFilter = new ParentFilter(config.getConfig("parent-filter"), geobaseTree)
  private val childrenFilter = new ChildrenFilter(config.getConfig("children-filter"), geobaseTree)
  private val ostrovokIdAmplifier = new OstrovokIdAmplifier(config.getConfig("ostrovok-id-amplifier"))
  private val hotelCountsFilter = new HotelCountsFilter(config.getConfig("hotel-counts-filter"))
  private val ostrovokSameNameParentDamper = new OstrovokSameNameParentDamper(config.getConfig("ostrovok-parent-damper"))

  private val matchers: Seq[RegionMatcher] = Seq(nameMatcher, hotelMatcher)
  private val filters: Seq[HypothesisFilter] = Seq(
    typeFilter,
    popularityAmplifier,
    coordsFilter,
    countryFilter,
    parentFilter,
    childrenFilter,
    ostrovokIdAmplifier,
    ostrovokSameNameParentDamper,
    hotelCountsFilter
  )

  private val absoluteCriteria = config.getDouble("criteria.absolute")
  private val relativeCriteria = config.getDouble("criteria.relative")

  def checkHypothesis(hypothesis: Hypothesis, partnerTree: PartnerTree,
                      mapping: Map[String, Iterable[Int]]): Hypothesis = {
    val modifiers = filters.map(_.apply(hypothesis, mapping, partnerTree))
    hypothesis.copy(confidence = hypothesis.confidence * modifiers.product)
  }

  private def alternativeTo(a: Hypothesis, b: Hypothesis): Boolean = {
    if (a.confidence * relativeCriteria >= b.confidence)
      if (a.partnerRegion.isDeparture == b.partnerRegion.isDeparture)
        if (a != b)
          return true
    false
  }

  def run(partnerTree: PartnerTree, mapping: Map[String, Iterable[Int]]): MatchingResults = {
    val all = for {
      matcher <- matchers
      hypothesis <- maxConfidence(matcher.createHypotheses(partnerTree))
    } yield hypothesis
    val grouped = multiplyConfidences(all)
    val filtered = grouped.map(checkHypothesis(_, partnerTree, mapping))

    var notResolvedHyps = mutable.Set.empty[Hypothesis]
    var droppedHyps = mutable.Set.empty[Hypothesis]
    def resolveDups(hyps: Seq[Hypothesis]): Option[Hypothesis] =
    {
      val mainHyp = hyps.maxBy(_.confidence)
      if (hyps.size > 1) {
        val alternatives = hyps.filter(alternativeTo(_, mainHyp))
        if (alternatives.nonEmpty) {
          notResolvedHyps ++= (alternatives :+ mainHyp)
          droppedHyps ++= hyps.diff(alternatives :+ mainHyp)
          return None
        }
      }
      droppedHyps ++= hyps.diff(Seq(mainHyp))
      if (mainHyp.confidence < absoluteCriteria) {
        notResolvedHyps += mainHyp
        return None
      }
      Some(mainHyp)
    }

    val matchedOk = filtered.
      toSeq.groupBy(_.yandexRegion).values.flatMap(resolveDups).
      toSeq.groupBy(_.partnerHeader).values.flatMap(resolveDups)
    MatchingResults(matchedOk, notResolvedHyps, droppedHyps)
  }

  private def combineWithGroupConfidence(hyps: Iterable[Hypothesis],
                                         rule: Iterable[Hypothesis] => Double): Iterable[Hypothesis] = {
    hyps.groupBy(h => (h.partnerHeader, h.yandexRegion)).map {
      case ((partnerHeader, yandexRegion), hypSeq) =>
        Hypothesis(yandexRegion, partnerHeader, rule(hypSeq))
    }
  }

  private def maxConfidence(hyps: Iterable[Hypothesis]): Iterable[Hypothesis] =
    combineWithGroupConfidence(hyps, _.map(_.confidence).max)

  private def multiplyConfidences(hyps: Iterable[Hypothesis]): Iterable[Hypothesis] =
    combineWithGroupConfidence(hyps, _.map(_.confidence + 1.0).product - 1.0)
}

case class MatchingResults(ok: Iterable[Hypothesis],
                           notResolved: Iterable[Hypothesis],
                           dropped: Iterable[Hypothesis] = Iterable.empty) {

  def all = ok ++ notResolved ++ dropped
}
