package ru.yandex.tours.geo.matching

import java.io.InputStream

import ru.yandex.extdata.common.meta.DataType
import ru.yandex.tours.extdata.{DataDefWithDependencies, DataTypes}
import ru.yandex.tours.geo.base.region.Tree
import ru.yandex.tours.geo.base.{Region, region}
import ru.yandex.tours.geo.mapping.GeoMappingShort
import ru.yandex.tours.geo.partners.{PartnerRegion, PartnerTrees}
import ru.yandex.tours.model.hotels.Partners
import ru.yandex.tours.model.hotels.Partners._
import ru.yandex.tours.util.IO
import ru.yandex.tours.util.parsing.{DoubleValue, IntValue, Tabbed}
import shapeless.{::, HNil}

import scala.collection.mutable

/**
  * Created by asoboll on 16.02.16.
  */
class HypothesesHolder(val values: Map[Partner, PartnerHypotheses]) {

  private val defaultMaxHypothesesForGeoId = 5

  def apply(partner: Partner): PartnerHypotheses =
    values.getOrElse(partner, new PartnerHypotheses(partner))

  def findAlternatives(hyp: Hypothesis): Iterator[Hypothesis] =
    findRelated(hyp).filterNot(_ == hyp)

  def findRelated(hyp: Hypothesis): Iterator[Hypothesis] =
    apply(hyp.partner).findRelated(hyp)

  def contains(hyp: Hypothesis): Boolean =
    apply(hyp.partner).findHypothesis(hyp.idForm).nonEmpty

  def findHypothesis(geoMappingShort: GeoMappingShort): Option[Hypothesis] =
    apply(geoMappingShort.partner).findHypothesis(geoMappingShort)

  def findHypothesis(partnerRegion: PartnerRegion, yaRegion: Region): Option[Hypothesis] =
    findHypothesis(GeoMappingShort(partnerRegion.partner, yaRegion.id, partnerRegion.id))

  /*def notResolved(partner: Partner, geoMapping: Iterable[GeoMappingShort], tree: region.Tree,
                  maxHypothesesForGeoId: Int = defaultMaxHypothesesForGeoId): Seq[Seq[Hypothesis]] = {
    val hyps = apply(partner).groupBy(_.yandexRegion.id)
    (for {
      geoId <- (hyps.keySet -- geoMapping.map(_.geoId).toSet).toSeq
    } yield {
      hyps(geoId).toSeq.sortBy(_.confidence).reverse.take(maxHypothesesForGeoId)
    }).sortBy(_.head.yandexRegion.position).reverse
  }*/

  def byPartnerRegion(region: PartnerRegion): Iterable[Hypothesis] =
    apply(region.partner).byPartnerRegion(region).toSeq.sortBy(_.confidence).reverse

  def byYandexRegion(region: Region, partner: Partner): Iterable[Hypothesis] =
    apply(partner).byYandexRegion(region, partner).toSeq.sortBy(_.confidence).reverse
}

object GeoMatchingHypotheses extends DataDefWithDependencies[HypothesesHolder, region.Tree :: PartnerTrees :: HNil]{
  override def dataType: DataType = DataTypes.geoMatchingHypotheses
  override def dependsOn: Set[DataType] = Set(DataTypes.regions, DataTypes.partnerRegions)

  def parse(is: InputStream, dependencies: Tree :: PartnerTrees :: HNil): HypothesesHolder = {
    val tree :: partnerTrees :: HNil = dependencies

    val builder: Map[Partner, (mutable.Buffer[region.Id], mutable.Buffer[String], mutable.Buffer[Double])] =
      Partners.values.toSeq.map { p =>
        p -> (mutable.ArrayBuffer.empty[region.Id], mutable.ArrayBuffer.empty[String], mutable.ArrayBuffer.empty[Double])
      }.toMap

    for {
      Tabbed(IntValue(partner), IntValue(geoId), partnerId, DoubleValue(confidence), modifiers) <- IO.readLines(is)
      partner <- Partners.getOpt(partner)
      buffers <- builder.get(partner)
    } {
      buffers._1 += geoId
      buffers._2 += partnerId
      buffers._3 += confidence
    }

    val hyps: Map[Partner, PartnerHypotheses] =
      for ((partner, (geoIds, partnerIds, confidences)) <- builder) yield {
        partner -> new PartnerHypotheses(tree, partnerTrees(partner), geoIds.toArray, partnerIds.toArray, confidences.toArray)
      }

    new HypothesesHolder(hyps)
  }

  def empty = new HypothesesHolder(Map.empty)
}
