package ru.yandex.tours.geo.matching

import com.typesafe.config.Config
import ru.yandex.extdata.common.meta.DataType
import ru.yandex.tours.extdata.DataTypes
import ru.yandex.tours.geo.base.region.Types
import ru.yandex.tours.geo.base.{Region, region}
import ru.yandex.tours.geo.mapping.GeoMappingShort
import ru.yandex.tours.geo.partners.{PartnerRegionHeader, PartnerTree, PartnerCategories, PartnerRegion}
import ru.yandex.tours.model.hotels.Partners
import ru.yandex.tours.model.hotels.Partners.Partner
import ru.yandex.tours.util.parsing.Tabbed

import scala.math._
import scala.util.Try

/**
  * Created by asoboll on 29.01.16.
  */
case class Hypothesis(yandexRegion: Region, partnerHeader: PartnerRegionHeader, confidence: Double = 0.0) {

  def partnerRegion: PartnerRegion = partnerHeader.partnerRegion

  def partner: Partner = partnerHeader.partner

  def partnerId: String = partnerHeader.id

  def fullPartnerId: (Partner, String) = (partner, partnerId)

  def hasType(dataType: DataType): Boolean = dataType match {
    case DataTypes.countries => yandexRegion.isCountry
    case DataTypes.departures => partnerRegion.isDeparture
    case DataTypes.cities => !hasType(DataTypes.countries)
    case _ => false
  }

  def idForm = GeoMappingShort(partner, yandexRegion.id, partnerId)

  def toTsv = Tabbed(partner.id, yandexRegion.id, partnerId, confidence, "")
}

trait HypothesisFilter {
  def apply(hypothesis: Hypothesis, mapping: Map[String, Iterable[Int]], partnerTree: PartnerTree): Double
}

trait TypeFilterMap {
  protected val yandexCategoryMap = Map(
    Types.Country -> PartnerCategories.Country,
    Types.OverseasLand -> PartnerCategories.Country,
    Types.Region -> PartnerCategories.Region,
    Types.FederalDistrict -> PartnerCategories.Region,
    Types.FederalSubject -> PartnerCategories.Region,
    Types.FederalSubjectRegion -> PartnerCategories.Region,
    Types.City -> PartnerCategories.Resort,
    Types.Village -> PartnerCategories.Resort,
    Types.Beach -> PartnerCategories.Resort,
    Types.SkiResort -> PartnerCategories.Resort,
    Types.MetaRegion -> PartnerCategories.RegionOrResort,
    Types.MetroStation -> PartnerCategories.Ignored,
    Types.MonorailStation -> PartnerCategories.Ignored,
    Types.SecondLevelCityDistrict -> PartnerCategories.Ignored,
    Types.Hidden -> PartnerCategories.Ignored,
    Types.Other -> PartnerCategories.Other,
    Types.Continent -> PartnerCategories.Ignored,
    Types.CityDistrict -> PartnerCategories.Unknown,
    Types.Airport -> PartnerCategories.Unknown,
    Types.RuralSettlement -> PartnerCategories.Unknown,
    Types.MultiPurpose -> PartnerCategories.Unknown
  )
}

class TypeFilter(config: Config) extends HypothesisFilter with TypeFilterMap {
  private val countryCoeff = config.getDouble("country-coeff")
  private val amplCoeff = config.getDouble("ampl-coeff")
  private val dampCoeff = config.getDouble("damp-coeff")

  private def confidenceModifier(yaCategory: PartnerCategories.Value, pCategory: PartnerCategories.Value): Double =
    (yaCategory, pCategory) match {
      case (PartnerCategories.Ignored, pCat) => 0.0
      case (yaCat, PartnerCategories.Ignored) => 0.0
      case (PartnerCategories.Other, pCat) => dampCoeff
      case (yaCat, PartnerCategories.Other) => dampCoeff
      case (PartnerCategories.Unknown, pCat) => 1.0
      case (yaCat, PartnerCategories.Unknown) => 1.0
      case (PartnerCategories.Country, PartnerCategories.Country) => countryCoeff
      case (PartnerCategories.Country, pCat) => dampCoeff
      case (yaCat, PartnerCategories.Country) => dampCoeff
      case (PartnerCategories.RegionOrResort, pCat) => amplCoeff
      case (yaCat, PartnerCategories.RegionOrResort) => amplCoeff
      case (yaCat, pCat) => if (yaCat == pCat) amplCoeff else dampCoeff
    }

  def apply(hypothesis: Hypothesis, mapping: Map[String, Iterable[Int]], partnerTree: PartnerTree): Double =
    yandexCategoryMap.get(hypothesis.yandexRegion.`type`).map { yaCategory =>
      confidenceModifier(yaCategory, hypothesis.partnerRegion.category)
    }.getOrElse(1.0)
}

class PopularityAmplifier(config: Config) extends HypothesisFilter {
  private val index = config.getDouble("scale-index")

  def apply(hypothesis: Hypothesis, mapping: Map[String, Iterable[Int]], partnerTree: PartnerTree): Double =
    math.pow(1.0 + hypothesis.yandexRegion.position, index)
}

class CoordsFilter(config: Config) extends HypothesisFilter {
  private val scale = config.getDouble("scale")
  private val identityCoeff = config.getDouble("identity-coeff")
  private val minCoeff = config.getDouble("min-coeff")
  private val kmInDegree = config.getDouble("km-in-degree")

  def apply(hypothesis: Hypothesis, mapping: Map[String, Iterable[Int]], partnerTree: PartnerTree): Double = {
    if (hypothesis.yandexRegion.isCountry || hypothesis.partnerRegion.category == PartnerCategories.Country)
      return 1.0
    val (pLat, pLon) = (toRadians(hypothesis.partnerRegion.latitude), toRadians(hypothesis.partnerRegion.longitude))
    val (yaLat, yaLon) = (toRadians(hypothesis.yandexRegion.latitude), toRadians(hypothesis.yandexRegion.longitude))
    if ((pLat, pLon) == (0.0, 0.0) || (yaLat, yaLon) == (0.0, 0.0))
      return 1.0
    val distance = kmInDegree * toDegrees(acos(sin(yaLat) * sin(pLat) + cos(yaLat) * cos(pLat) * cos(pLon - yaLon)))
    max(minCoeff, identityCoeff / sqrt(1.0 + distance * 3.0 / scale))
  }
}

class ChildrenFilter(config: Config, geobaseTree: region.Tree) extends HypothesisFilter {
  private val amplCoeff = config.getDouble("ampl-coeff")
  private val dampCoeff = config.getDouble("damp-coeff")

  def apply(hypothesis: Hypothesis, mapping: Map[String, Iterable[Int]], partnerTree: PartnerTree): Double = {
    val mappedViaChildrenIds = for {
      pChild <- partnerTree.children(hypothesis.partnerRegion)
      mappedChild <- mapping.getOrElse(pChild.id, Iterable.empty)
      possibleMapped <- geobaseTree.findParents(mappedChild)
      if possibleMapped.id != mappedChild
    } yield possibleMapped.id
    if (mappedViaChildrenIds.nonEmpty)
      if (mappedViaChildrenIds.contains(hypothesis.yandexRegion.id)) amplCoeff
      else dampCoeff
    else 1.0
  }
}

class CountryFilter(config: Config, geobaseTree: region.Tree) extends HypothesisFilter {
  private val amplCoeff = config.getDouble("ampl-coeff")
  private val dampCoeff = config.getDouble("damp-coeff")

  def apply(hypothesis: Hypothesis, mapping: Map[String, Iterable[Int]], partnerTree: PartnerTree): Double = {
    if ((!hypothesis.yandexRegion.isCountry) && (hypothesis.partnerRegion.category != PartnerCategories.Country)) {
      for {
        partnerCountryId <- hypothesis.partnerRegion.countryId
        yandexCountry <- geobaseTree.country(hypothesis.yandexRegion)
        possibleCountries <- mapping.get(partnerCountryId)
      } {
        if (possibleCountries.exists(_ == yandexCountry.id)) return amplCoeff
        else return dampCoeff
      }
    }
    1.0
  }
}

class ParentFilter(config: Config, geobaseTree: region.Tree) extends HypothesisFilter {
  private val amplCoeff = config.getDouble("ampl-coeff")
  private val dampCoeff = config.getDouble("damp-coeff")

  def apply(hypothesis: Hypothesis, mapping: Map[String, Iterable[Int]], partnerTree: PartnerTree): Double = {
    val yandexParentsIds = geobaseTree.findParents(hypothesis.yandexRegion).map(_.id) - hypothesis.yandexRegion.id
    val mappedParentIds = (for {
      partnerParent <- partnerTree.parent(hypothesis.partnerRegion)
      if hypothesis.partnerRegion.countryId.map(_ != partnerParent.id).getOrElse(true) // don't want to duplicate country filter
      possibleParents <- mapping.get(partnerParent.id)
    } yield possibleParents).getOrElse(Iterable.empty).toSet
    if (mappedParentIds.nonEmpty && yandexParentsIds.nonEmpty)
      if ((mappedParentIds & yandexParentsIds).nonEmpty) amplCoeff
      else dampCoeff
    else 1.0
  }
}

class OstrovokIdAmplifier(config: Config) extends HypothesisFilter {
  private val amplCoeff = config.getDouble("ampl-coeff")

  def apply(hypothesis: Hypothesis, mapping: Map[String, Iterable[Int]], partnerTree: PartnerTree): Double = {
    if (hypothesis.partner == Partners.ostrovok || hypothesis.partner == Partners.ostrovokv3)
      if (Try(hypothesis.partnerId.toInt).toOption.exists(id => (id >= 965000000) || (id < 600000)))
        return amplCoeff
    1.0
  }
}

class HotelCountsFilter(config: Config) extends HypothesisFilter {
  private val dampCoeff = config.getDouble("damp-coeff")
  private val index = config.getDouble("scale-index")
  private def amplCoeff(n: Int): Double = math.pow(n, index)

  def apply(hypothesis: Hypothesis, mapping: Map[String, Iterable[Int]], partnerTree: PartnerTree): Double = {
    if (hypothesis.partnerRegion.category != PartnerCategories.Country)
      hypothesis.partnerRegion.hotelsCount match {
        case 0 => dampCoeff
        case n => amplCoeff(n)
      }
    else 1.0
  }
}

class OstrovokSameNameParentDamper(config: Config) extends HypothesisFilter {
  private val dampCoeff = config.getDouble("damp-coeff")

  def apply(hypothesis: Hypothesis, mapping: Map[String, Iterable[Int]], partnerTree: PartnerTree): Double = {
    if (hypothesis.partner == Partners.ostrovok || hypothesis.partner == Partners.ostrovokv3)
      if (hypothesis.partnerRegion.category != PartnerCategories.Country)
        if (partnerTree.children(hypothesis.partnerRegion).exists(_.getName == hypothesis.partnerRegion.getName))
          return dampCoeff
    1.0
  }
}