package ru.yandex.tours.geo.matching

import com.typesafe.config.Config
import ru.yandex.tours.geo.base.region
import ru.yandex.tours.geo.partners.{PartnerCategories, PartnerTree}
import ru.yandex.tours.hotels.HotelsIndex
import ru.yandex.tours.model.hotels.Partners.Partner
import ru.yandex.tours.util.Logging
import ru.yandex.tours.util.collections.Bag

import scala.collection.mutable

/**
  * Created by asoboll on 21.02.17.
  */
class RegionMatcherByHotelsIndex(config: Config,
                                 regionTree: region.Tree,
                                 hotelsIndex: HotelsIndex) extends RegionMatcher with Logging {
  private val oneHotelWeight = config.getDouble("one-hotel-weight")
  private val index = config.getDouble("scale-index")

  private def weight(hotelCount: Int): Double = Math.pow(hotelCount, index) * oneHotelWeight

  private def shouldApply(partner: Partner): Boolean = {
    // todo: should be applied to all partners ?
    true
  }

  private lazy val partner2Bag: scala.collection.Map[Partner, Bag[(String, Int)]] = {
    val bags = new mutable.HashMap[Partner, Bag[(String, Int)]]()
    for {
      hotel <- hotelsIndex.hotels
      partnerInfo <- hotel.partnerIds
      partnerRegionId <- partnerInfo.partnerRegionId
    } bags.getOrElseUpdate(partnerInfo.partner, new Bag) += ((partnerRegionId, hotel.geoId))
    bags
  }

  override def createHypotheses(partnerTree: PartnerTree): Iterable[Hypothesis] = {
    val partner = partnerTree.partner
    if (!shouldApply(partner)) return Iterable.empty[Hypothesis]

    val bag = partner2Bag.getOrElse(partner, new Bag)
    log.debug(s"Collected ${bag.size} region pairs for partner $partner")

    // sometimes hotel is in some partner region,
    // but we should match with its parent, because yandex geobase is not detailed enough
    val parentBag = new Bag[(String, Int)]
    for {
      ((partnerRegionId, yandexId), count) <- bag.toMap
      parent <- partnerTree.parent(partnerRegionId)
    } parentBag += ((parent.id, yandexId), count / 3)
    bag ++= parentBag

    for {
      ((partnerRegionId, yandexId), count) <- bag.toMap
      yaRegion <- regionTree.region(yandexId)
      if !yaRegion.isCountry
      partnerRegion <- partnerTree.header(partnerRegionId)
      if partnerRegion.partnerRegion.category != PartnerCategories.Country
    } yield Hypothesis(yaRegion, partnerRegion, weight(count))
  }
}
