package ru.yandex.tours

import java.lang.Math._

import ru.yandex.tours.model.geo.MapObject

/* @author berkut@yandex-team.ru */

package object geo {

  val EARTH_RADIUS_IN_KM = 6371.0088

  private def sqr(a: Double) = a * a

  /**
   * Use ellipsoidal Earth projected to a plane for calculate geo distance.
   * Works precisely for distance up to 475 km. <br/>
   * See <a href="http://en.wikipedia.org/wiki/Geographical_distance">wikipedia</a>
   *
   * @param lat1Degree  latitude of the first point in degrees
   * @param lon1Degree longitude of the first point in degrees
   * @param lat2Degree  latitude of the second point in degrees
   * @param lon2Degree longitude of the second point in degrees
   * @return distance between specified points in kilometers
   */
  def distanceInKm(lat1Degree: Double, lon1Degree: Double, lat2Degree: Double, lon2Degree: Double): Double = {
    val deltaLat = abs(lat1Degree - lat2Degree)
    val deltaLong = abs(lon1Degree - lon2Degree)
    val meanLat = (toRadians(lat1Degree) + toRadians(lat2Degree)) / 2
    val K1 = 111.13209 - 0.56605 * cos(2 * meanLat) + 0.0012 * cos(4 * meanLat)
    val K2 = 111.41513 * cos(meanLat) - 0.09455 * cos(3 * meanLat) + 0.00012 * cos(5 * meanLat)
    sqrt(sqr(K1 * deltaLat) + sqr(K2 * deltaLong))
  }

  def distanceInKm(point1: MapObject, point2: MapObject): Double =
    distanceInKm(point1.latitude, point1.longitude, point2.latitude, point2.longitude)

  private def toPoint(lat: Double, lon: Double) = {
    val r = EARTH_RADIUS_IN_KM
    val latRad: Double = toRadians(lat)
    val coslat = Math.cos(latRad)
    val sinlat = Math.sin(latRad)
    val lonRad = toRadians(lon)
    val coslon = Math.cos(lonRad)
    val sinlon = Math.sin(lonRad)
    Point(r * coslat * coslon, r * coslat * sinlon, r * sinlat)
  }

  private case class Point(x: Double, y: Double, z: Double) {
    def -(other: Point): Point = Point(x - other.x, y - other.y, z - other.z)

    def *(other: Point): Point = Point(y * other.z - z * other.y, z * other.x - x * other.z, x * other.y - y * other.x)

    def norm: Double = Math.sqrt(x * x + y * y + z * z)

    def dot(other: Point): Double = x * other.x + y * other.y + z * other.z
  }

  private case class Line(a: Point, b: Point) {
    def distanceTo(other: Point): Double = ((other - a) * (other - b)).norm / (b - a).norm
  }

  def distanceToSegment(lat1: Double, lon1: Double, lat2: Double, lon2: Double, lat3: Double, lon3: Double): Double = {
    val from = toPoint(lat1, lon1)
    val to = toPoint(lat2, lon2)
    val point = toPoint(lat3, lon3)
    if (((to - from) dot (point - from)) > 0 && ((from - to) dot (point - to)) > 0) {
      // Point is between normals
      Line(from, to).distanceTo(point)
    } else {
      Math.min(distanceInKm(lat1, lon1, lat3, lon3), distanceInKm(lat2, lon2, lat3, lon3))
    }
  }

}
