package ru.yandex.tours.geo.partners

import java.io._
import java.nio.{ByteBuffer, MappedByteBuffer}

import com.google.protobuf.ByteString
import ru.yandex.tours.model.hotels.Partners
import ru.yandex.tours.model.hotels.Partners.Partner
import ru.yandex.tours.util.collections.ByteBufferMappedMap.{MappedMap, MappedMapWriter}
import ru.yandex.tours.util.io.ByteBuffers
import ru.yandex.tours.util.{IO, Logging}

import scala.concurrent.duration.FiniteDuration
import scala.util.Try
import scala.util.control.NonFatal

/**
  * Created by asoboll on 10.03.17.
  */
class MappedPartnerTree(partner: Partner,
                        index: Map[String, ExtendedPartnerRegion],
                        buffer: Option[MappedByteBuffer],
                        file: Option[File])
  extends PartnerTree(partner, index) with Closeable {

  def this(tree: PartnerTree) = this(tree.partner, tree.index, None, None)

  def load() = buffer.foreach(_.load())

  def delete() = file.foreach(_.delete())

  def close(): Unit = buffer.foreach(ByteBuffers.unmap)

  def closeAfter(interval: FiniteDuration): Unit = {
    IO.closeAfter(this, interval)
  }
}

object MappedPartnerTree extends Logging {
  private val formatVersion = 1
  private val headerSize = 3 * Integer.BYTES

  implicit val serializeString = (s: String) => s.getBytes
  implicit val deserializeString = (bb: ByteBuffer) => new String(ByteString.copyFrom(bb).toByteArray)
  implicit val serializeRegion = (r: ExtendedPartnerRegion) => ExtendedPartnerRegion.serialize(r)
  implicit val deserializeRegion = (bb: ByteBuffer) => ExtendedPartnerRegion.deserialize(ByteString.copyFrom(bb).toByteArray).get

  val empty: MappedPartnerTree = new MappedPartnerTree(Partners.unknown, Map.empty, None, None)

  def fromBuffer(buffer: ByteBuffer): MappedPartnerTree = {
    Try {
      val version = buffer.getInt
      require(version == formatVersion, s"Unknown format version: $version")
      val partner = Partners(buffer.getInt)
      val size = buffer.getInt
      val map = MappedMap.from[String, ExtendedPartnerRegion](buffer.slice)
      require(size == -1 || size == map.size, s"Record count ${map.size} doesnt match header: $size")
      new MappedPartnerTree(partner, map, None, None)
    }.recover {
      case NonFatal(t) =>
        log.error("Cannot parse partner regions tree", t)
        MappedPartnerTree.empty
    }.get
  }

  def fromFile(file: File): MappedPartnerTree = {
    val buffer = ByteBuffers.mmap(file)
    val tree = fromBuffer(buffer)
    new MappedPartnerTree(tree.partner, tree.index, Some(buffer), Some(file))
  }

  def writeTo(os: OutputStream, partnerTree: PartnerTree): Unit = {
    val mapWriter = new MappedMapWriter[String, ExtendedPartnerRegion](os)

    val header = ByteBuffer.allocate(headerSize)
      .putInt(formatVersion)
      .putInt(partnerTree.partner.id)
      .putInt(partnerTree.size)
    os.write(header.array())

    mapWriter.write(partnerTree.index.toSeq.sortBy(_._1))

    mapWriter.finish()
    os.close()
  }

  def writeTo(file: File, partnerTree: PartnerTree): Unit = {
    val os = new BufferedOutputStream(new FileOutputStream(file))
    writeTo(os, partnerTree)
  }
}
