package ru.yandex.tours.geo.partners

import java.io.InputStream

import ru.yandex.common.util.collections.Bag
import ru.yandex.extdata.common.exception.{DataIsNotLoadedException, UnableToLoadDataException}
import ru.yandex.extdata.common.meta.DataType
import ru.yandex.extdata.common.service.ExtDataService
import ru.yandex.tours.extdata.{DataDef, DataTypes}
import ru.yandex.tours.geo.partners.PartnerRegionHotelCounts.CountsMap
import ru.yandex.tours.model.hotels.HotelsHolder.{PartnerHotel, RawPartnerHotel}
import ru.yandex.tours.model.hotels.PartnerInfo
import ru.yandex.tours.model.hotels.Partners.Partner
import ru.yandex.tours.util.parsing.{IntValue, Tabbed}
import ru.yandex.tours.util.{IO, Logging}

import scala.collection.JavaConversions._
import scala.collection.mutable

/**
  * Created by asoboll on 27.02.16.
  */
class PartnerRegionHotelCounts(region2count: CountsMap) {
  def apply(partner: Int, partnerId: String): Int = region2count.get(partner).fold(0)(_.get(partnerId))

  def apply(partner: Partner, partnerId: String): Int = apply(partner.id, partnerId)

  def apply(partnerRegion: PartnerRegion): Int = apply(partnerRegion.partner.id, partnerRegion.id)

  def toByteArray: Array[Byte] =
    IO.printBytes { pw =>
      for {
        (partner, partnerMap) <- region2count
        pair <- partnerMap.asDescSortedList()
      } pw.println(Tabbed(partner, pair.getFirst, pair.getSecond))
    }

  def isEmpty: Boolean = region2count.isEmpty
}

object PartnerRegionHotelCounts extends DataDef[PartnerRegionHotelCounts] with Logging {
  private type CountsMap = mutable.Map[Int, Bag[String]]
  def newCountsMap: mutable.Map[Int, Bag[String]] = mutable.Map.empty[Int, Bag[String]]

  def dataType: DataType = DataTypes.partnerRegionHotelCounts

  override def from(extDataService: ExtDataService): PartnerRegionHotelCounts = {
    try {
      super.from(extDataService)
    } catch {
      case _: DataIsNotLoadedException | _: UnableToLoadDataException =>
        log.warn(s"No previous ${dataType.getName}. Using empty one.")
        PartnerRegionHotelCounts.empty
    }
  }

  def parse(is: InputStream): PartnerRegionHotelCounts = {
    val data = for (Tabbed(IntValue(partner), partnerId, IntValue(hotelsCount)) <- IO.readLines(is))
      yield (partner, partnerId, hotelsCount)
    apply(data)
  }

  private def apply(data: TraversableOnce[(Int, String, Int)]): PartnerRegionHotelCounts = {
    val region2count = newCountsMap
    for ((partner, partnerId, hotelsCount) <- data)
      countRegion(region2count)(partner, partnerId, hotelsCount)
    new PartnerRegionHotelCounts(region2count)
  }

  private def countRegion(region2count: CountsMap)
                         (partner: Int, partnerId: String, hotelsCount: Int) = {
    val bag = region2count.getOrElseUpdate(partner, Bag.newHashBag[String])
    bag.add(partnerId, hotelsCount)
  }

  def countHotel(region2count: CountsMap)(rawHotel: RawPartnerHotel): Unit = {
    if (rawHotel.hasPartner && rawHotel.hasRegionId) {
      countRegion(region2count)(rawHotel.getPartner, rawHotel.getRegionId, 1)
    }
  }

  def countPartnerInfo(region2count: CountsMap)(partnerInfo: PartnerInfo): Unit = {
    partnerInfo.partnerRegionId.foreach { partnerRegionId =>
      countRegion(region2count)(partnerInfo.partner.id, partnerRegionId, 1)
    }
  }

  def from(hotels: TraversableOnce[(Int, PartnerHotel)]): PartnerRegionHotelCounts = {
    val region2count = newCountsMap
    for {
      (id, hotel) <- hotels
    } countHotel(region2count)(hotel.getRawHotel)
    new PartnerRegionHotelCounts(region2count)
  }

  def empty: PartnerRegionHotelCounts = new PartnerRegionHotelCounts(newCountsMap)
}