package ru.yandex.tours.geo.partners

import ru.yandex.tours.model.hotels.Partners
import ru.yandex.tours.model.{Languages, LocalizedString}
import ru.yandex.tours.util.IO
import ru.yandex.tours.util.parsing.{DoubleValue, IntValue, Tabbed}
import ru.yandex.vertis.curator.recipes.map.ValueSerializer

import scala.util.Try

/**
  * Created by asoboll on 22.03.16.
  */
object PartnerRegionParser extends ValueSerializer[PartnerRegion]  {
  def writeString(r: PartnerRegion): String = Tabbed(
    r.partner,
    r.id,
    r.name.get(Languages.ru).getOrElse(""),
    r.name.get(Languages.en).getOrElse(""),
    r.category,
    r.origType.getOrElse(""),
    if (r.isDeparture) "departure" else "",
    r.longitude,
    r.latitude,
    r.hotelsCount,
    r.synonyms.mkString("#"),
    r.parentId.getOrElse(""),
    r.countryId.getOrElse("")
  )

  def serialize(r: PartnerRegion): Array[Byte] = {
    IO.printBytes { pw => pw.println(writeString(r)) }
  }

  def parse(s: String, hotelCounts: PartnerRegionHotelCounts = PartnerRegionHotelCounts.empty): Try[PartnerRegion] = {
    s match {
      case Tabbed(partnerName, id, ruName, enName, geoType, origType, isDeparture,
      strLon, strLat, hotelsCount, synonyms, parentId, countryId) =>
        Try {
          val partner = Partners.withName(partnerName)
          val names =
            for {
              (lang, string) <- Seq(Languages.ru -> ruName, Languages.en -> enName)
              if string.nonEmpty
            } yield lang -> string
          val internalHotelsCount = hotelCounts(partner, id)
          val internalHotelsCountOpt = if (internalHotelsCount == 0) None else Some(internalHotelsCount)

          PartnerRegion(
            partner,
            id,
            LocalizedString(names.toMap),
            PartnerCategories.withName(geoType),
            if (origType.isEmpty) None else Some(origType),
            isDeparture = isDeparture == "departure",
            DoubleValue(strLon).getOrElse(0.0),
            DoubleValue(strLat).getOrElse(0.0),
            (internalHotelsCountOpt orElse IntValue(hotelsCount)).getOrElse(0),
            synonyms.split("#").toSet,
            if (parentId.isEmpty) None else Some(parentId),
            if (countryId.isEmpty) None else Some(countryId)
          )
        }
    }
  }

  def vashotelHack(region: PartnerRegion, hotelCounts: PartnerRegionHotelCounts): Option[PartnerRegion] = Option {
    if (region.partner == Partners.vashotel) {
      val origId = region.originalId
      if (region.category == PartnerCategories.Country) {
        if (origId != "1") return None
        region.copy(id = "1_dup_Country_0")
      } else {
        if ((!hotelCounts.isEmpty) && (hotelCounts(region.partner, origId) == 0)) return None
        region.copy(id = origId,
          parentId = region.parentId.map(_ + "_dup_Country_0"),
          countryId = region.countryId.map(_ + "_dup_Country_0"))
      }
    } else {
      region
    }
  }

  def deserialize(bytes: Array[Byte]): Try[PartnerRegion] = deserialize(new String(bytes))

  def deserialize(s: String): Try[PartnerRegion] = parse(s)
}