package ru.yandex.tours.geo.partners

import ru.yandex.tours.geo.base.TreeBase
import ru.yandex.tours.model.hotels.Partners.Partner

/**
  * Created by asoboll on 29.01.16.
  */
class PartnerTree(val partner: Partner,
                  val index: Map[String, ExtendedPartnerRegion])
  extends TreeBase[String, PartnerRegion] {

  lazy val rootIds: Seq[String] = extendedRegions.filter(_.parentId.isEmpty).map(_.id).toSeq

  def roots: Seq[PartnerRegion] = rootIds.flatMap(region(_).toSeq)

  def size: Int = index.size

  def regions: Iterable[PartnerRegion] = extendedRegions.map(_.region).toIterable

  def extendedRegions: Iterator[ExtendedPartnerRegion] = index.valuesIterator

  def header(partnerId: String): Option[PartnerRegionHeader] =
    if (index.contains(partnerId)) Some(PartnerRegionHeader(partnerId, this))
    else None

  def regionHeaders: Iterable[PartnerRegionHeader] = index.keys.map(PartnerRegionHeader(_, this))

  private def get[T](id: String, param: ExtendedPartnerRegion => T): Option[T] = index.get(id).map(param)
  private def get2[T](id: String, param: ExtendedPartnerRegion => Option[T]): Option[T] = index.get(id).flatMap(param)

  def region(id: String): Option[PartnerRegion] = get(id, _.region)
  def region(r: PartnerRegion): Option[PartnerRegion] = region(r.id)

  def parent(id: String): Option[PartnerRegion] = get2(id, _.parentId).flatMap(region)
  def parent(r: PartnerRegion): Option[PartnerRegion] = parent(r.id)

  def children(id: String): List[PartnerRegion] = get(id, _.childrenIds).toList.flatten.flatMap(region(_).toList)
  def children(r: PartnerRegion): List[PartnerRegion] = children(r.id)

  def pathToRoot(id: String): List[PartnerRegion] = {
    region(id) match {
      case Some(region) => pathToRoot(region)
      case None => Nil
    }
  }

  def pathToRoot(region: PartnerRegion): List[PartnerRegion] = {
    parent(region) match {
      case Some(parent) if parent.id != region.id => region :: pathToRoot(parent)
      case Some(parent) => sys.error(s"Region $parent is parent of itself")
      case None => region :: Nil
    }
  }

  def traverseWithLevel(region: PartnerRegion): Traversable[(PartnerRegion, Int)] =
    new Traversable[(PartnerRegion, Int)] {
      private def traverse[U](r: PartnerRegion, level: Int, f: ((PartnerRegion, Int)) => U): Unit = {
        f(r, level)
        if (r == region) {
          for (child <- children(r)) {
            f(child, level + 1)
          }
        }
      }

      override def foreach[U](f: ((PartnerRegion, Int)) => U): Unit =
        if (rootIds.contains(region.id)) {
          traverse(region, 0, f)
          roots.filterNot(_ == region).foreach(f(_, 0))
        } else {
          pathToRoot(region).reverse.zipWithIndex.foreach { case (r, level) => traverse(r, level, f) }
        }
    }
}
