package ru.yandex.tours.hotels

import ru.yandex.tours.geo.base.region.Tree
import ru.yandex.tours.model.MapRectangle
import ru.yandex.tours.model.filter.HotelFilter
import ru.yandex.tours.model.geo.MapObject
import ru.yandex.tours.model.hotels.Hotel
import ru.yandex.tours.model.hotels.Partners.Partner
import ru.yandex.tours.model.util.Paging

/**
  * Filters hotels without corresponding regions
  */
class FilteringHotelsIndex(hotelsIndex: HotelsIndex, tree: Tree) extends HotelsIndex {

  override def hotels: Iterator[Hotel] = hotelsIndex.hotels.filter(isValid)

  override def size: Int = hotelsIndex.size

  override def getHotelsById(ids: Iterable[Int]): Map[Int, Hotel] = hotelsIndex.getHotelsById(ids).filter {
    case (_, hotel) => isValid(hotel)
  }

  override def filter(ids: Iterable[Int],
                      filters: Iterable[HotelFilter],
                      mr: Option[MapRectangle]): Set[Int] = {
    hotelsIndex.filter(ids, filters, mr)
  }

  override def near(hotel: MapObject, maxSize: Int, filters: HotelFilter*): Iterator[Hotel] = {
    hotelsIndex.near(hotel, maxSize, filters : _*).filter(isValid)
  }

  override def getHotelById(id: Int): Option[Hotel] = hotelsIndex.getHotelById(id).filter(isValid)

  override def topInRegion(geoId: Int,
                           maxSize: Int,
                           filters: HotelFilter*): Iterator[Hotel] = {
    hotelsIndex.topInRegion(geoId, maxSize, filters: _*).filter(isValid)
  }

  override def hotelsCountInRegion(geoId: Int): Int = hotelsIndex.hotelsCountInRegion(geoId)

  override def inRectangle(mapInfo: MapRectangle, maxSize: Int, filters: HotelFilter*): Seq[Hotel] = {
    hotelsIndex.inRectangle(mapInfo, maxSize, filters: _*).filter(isValid)
  }

  override def getHotels(ids: Iterable[Int],
                         filters: Iterable[HotelFilter],
                         mr: Option[MapRectangle]): Map[Int, Hotel] = {
    hotelsIndex.getHotels(ids, filters, mr).filter {
      case (_, hotel) => isValid(hotel)
    }
  }

  override def getHotel(partner: Partner, id: String): Option[Hotel] = {
    hotelsIndex.getHotel(partner, id).filter(isValid)
  }

  override def getHotels(page: Paging, filters: HotelFilter*): Seq[Hotel] =
    hotelsIndex.getHotels(page, filters: _*).filter(isValid)

  override def getHotelsSorted(page: Paging, filters: HotelFilter*): Seq[Hotel] =
    hotelsIndex.getHotelsSorted(page, filters: _*).filter(isValid)

  override def count(filters: HotelFilter*): Int = hotelsIndex.count(filters: _*)

  private def isValid(hotel: Hotel): Boolean = {
    tree.region(hotel.geoId).nonEmpty
  }

  override def getHotelsCount(ids: Iterable[Int],
                              filters: Iterable[HotelFilter],
                              mapRectangle: Option[MapRectangle]): Int = {
    hotelsIndex.getHotelsCount(ids, filters, mapRectangle)
  }
}
