package ru.yandex.tours.hotels

import java.io.InputStream

import it.unimi.dsi.fastutil.ints.{Int2DoubleOpenHashMap, Int2IntOpenHashMap}
import ru.yandex.extdata.common.exception.{UnableToLoadDataException, DataIsNotLoadedException}
import ru.yandex.extdata.common.meta.DataType
import ru.yandex.extdata.common.service.ExtDataService
import ru.yandex.tours.extdata.{DataDef, DataTypes}
import ru.yandex.tours.model.hotels.Hotel
import ru.yandex.tours.util.{Logging, IO}
import ru.yandex.tours.util.parsing.{DoubleValue, IntValue, Tabbed}

class HotelRatings(hotelId2rating: Int2DoubleOpenHashMap,
                   hotelId2reviews: Int2IntOpenHashMap,
                   hotelId2Visits: Int2IntOpenHashMap) {

  def getRating(hotelId: Int): Double = hotelId2rating.get(hotelId)
  def getReviewsCount(hotelId: Int): Int = hotelId2reviews.get(hotelId)

  def getVisits(hotelId: Int): Int = hotelId2Visits.get(hotelId)

  def getRelevance(hotelId: Int): Double = {
    val reviewsCount = getReviewsCount(hotelId)
    val rating = getRating(hotelId)
    Hotel.relevance(rating, reviewsCount)
  }

  def copy(): HotelRatings = new HotelRatings(hotelId2rating, hotelId2reviews, hotelId2Visits)
}

object HotelRatings extends DataDef[HotelRatings] with Logging {

  override def dataType: DataType = DataTypes.hotelsRatings

  override def from(extDataService: ExtDataService): HotelRatings = {
    try {
      super.from(extDataService)
    } catch {
      case _: DataIsNotLoadedException | _: UnableToLoadDataException =>
        log.warn(s"No previous ${dataType.getName}. Using empty one.")
        HotelRatings.empty
    }
  }

  override def parse(is: InputStream): HotelRatings = {
    val hotelId2Rating = new Int2DoubleOpenHashMap()
    val hotelId2Reviews = new Int2IntOpenHashMap()
    val hotelId2Visits = new Int2IntOpenHashMap()

    IO.readLines(is).foreach {
      case Tabbed(IntValue(hotelId), DoubleValue(rating), IntValue(reviewsCount)) ⇒
        hotelId2Rating.put(hotelId, rating)
        hotelId2Reviews.put(hotelId, reviewsCount)
      case Tabbed(IntValue(hotelId), DoubleValue(rating), IntValue(reviewsCount), IntValue(visits)) ⇒
        hotelId2Rating.put(hotelId, rating)
        hotelId2Reviews.put(hotelId, reviewsCount)
        hotelId2Visits.put(hotelId, visits)
    }

    new HotelRatings(hotelId2Rating, hotelId2Reviews, hotelId2Visits)
  }

  def parseWithConv(is: InputStream, conv: Map[String, Int]): HotelRatings = {
    val hotelId2Rating = new Int2DoubleOpenHashMap()
    val hotelId2Reviews = new Int2IntOpenHashMap()

    IO.readLines(is).foreach {
      case Tabbed(hotelIdStr, DoubleValue(rating), IntValue(reviewsCount)) =>
        val hotelId = conv.get(hotelIdStr)
        if (hotelId.isDefined) {
          hotelId2Rating.put(hotelId.get, rating)
          hotelId2Reviews.put(hotelId.get, reviewsCount)
        }
    }

    new HotelRatings(hotelId2Rating, hotelId2Reviews, new Int2IntOpenHashMap(0))
  }

  def apply(hotels: TraversableOnce[Hotel]): HotelRatings = {
    val hotelId2Rating = new Int2DoubleOpenHashMap()
    val hotelId2Reviews = new Int2IntOpenHashMap()

    for (hotel <- hotels) {
      hotelId2Rating.put(hotel.id, hotel.rating)
      hotelId2Reviews.put(hotel.id, hotel.reviewsCount)
    }
    new HotelRatings(hotelId2Rating, hotelId2Reviews, new Int2IntOpenHashMap(0))
  }

  def apply(hotelsIndex: HotelsIndex): HotelRatings = apply(hotelsIndex.hotels)

  def empty: HotelRatings = {
    new HotelRatings(new Int2DoubleOpenHashMap(0), new Int2IntOpenHashMap(0), new Int2IntOpenHashMap(0))
  }
}
