package ru.yandex.tours.hotels

import java.io.{File, FileOutputStream, InputStream, PrintWriter}
import java.util.zip.{GZIPInputStream, GZIPOutputStream}

import ru.yandex.extdata.common.service.ExtDataService
import ru.yandex.tours.extdata.{DataTypes, ExtDataUpdateChecker}
import ru.yandex.tours.geo.base.region.Tree
import ru.yandex.tours.hotels.HotelsExport.Fields
import ru.yandex.tours.hotels.HotelsExport.Fields.Field
import ru.yandex.tours.model.Languages
import ru.yandex.tours.model.hotels.{Hotel, Partners}
import ru.yandex.tours.util.IO
import ru.yandex.tours.util.io.StringIteratorInputStream
import ru.yandex.tours.util.parsing.{AtSKV, Tabbed}

object HotelsExport {

  object Fields extends Enumeration {
    type Field = Value

    val ID = Value(0, "id")
    val NAME = Value(1, "name")
    val RU_NAME = Value(2, "ru_name")
    val EN_NAME = Value(3, "en_name")
    val LANGED_NAME = Value(4, "langed_name")
    val STARS = Value(5, "stars")
    val FEATURES = Value(6, "features")
    val LATITUDE = Value(7, "latitude")
    val LONGITUDE = Value(8, "longitude")
    val RATING = Value(9, "rating")
    val REVIEW_COUNT = Value(10, "review_count")
    val PARTNER_IDS = Value(11, "partner_ids")
    val TOURS_SEARCH = Value(12, "tours_search")
    val HOTEL_SEARCH = Value(13, "rooms_search")
    val REGION_ID = Value(14, "region_id")
    val COUNTRY_ID = Value(15, "country_id")
    val URL = Value(16, "url")
    val PHONES = Value(17, "phones")
    val BOOKING_ID = Value(18, "booking_id")

    val sortedValues = values.toSeq.sortBy(_.id)
  }

  private def extract(field: Field, hotel: Hotel, tree: Tree) = {
    val result = field match {
      case Fields.ID => hotel.id
      case Fields.NAME => hotel.name(Languages.ru)
      case Fields.RU_NAME => hotel.name(Languages.ru)
      case Fields.EN_NAME => hotel.name(Languages.en)
      case Fields.LANGED_NAME => AtSKV(hotel.name.values.map { case (lang, value) => lang.toString -> value })
      case Fields.STARS => hotel.star.id
      case Fields.FEATURES => AtSKV(hotel.features.flatMap(feature => feature.valueAsSet.map(value => feature.feature.name -> value.toString)))
      case Fields.LATITUDE => hotel.latitude
      case Fields.LONGITUDE => hotel.longitude
      case Fields.RATING => hotel.rating
      case Fields.REVIEW_COUNT => hotel.reviewsCount
      case Fields.PARTNER_IDS => AtSKV(hotel.partnerIds.map(pi => pi.partner.toString -> pi.id))
      case Fields.TOURS_SEARCH => hotel.toursSearchAvailable
      case Fields.HOTEL_SEARCH => hotel.roomsSearchAvailable
      case Fields.REGION_ID => hotel.geoId
      case Fields.COUNTRY_ID => tree.country(hotel.geoId).fold("")(_.id.toString)
      case Fields.URL => hotel.url.getOrElse("")
      case Fields.PHONES => hotel.phones.mkString("#")
      case Fields.BOOKING_ID => hotel.partnerIds.find(_.partner == Partners.booking).fold("")(_.id)
    }
    result.toString
  }

  def prepare(hotels: TraversableOnce[Hotel], tree: Tree): File = {
    val file = IO.newTempFile("hotels_export", "tmp.gz")
    IO.using(new PrintWriter(new GZIPOutputStream(new FileOutputStream(file)))) { pw ⇒
      hotels.foreach { hotel =>
        pw.println(Tabbed(Fields.sortedValues.map(extract(_, hotel, tree)): _*))
      }
    }
    file
  }
}

class HotelsExport(extDataLoader: ExtDataService, extDataUpdateChecker: ExtDataUpdateChecker) {
  extDataUpdateChecker.subscribe(DataTypes.hotelsExport)(())

  def read(fields: Iterable[Field], fits: (Fields.Value, String) => Boolean): InputStream = {
    val is = new GZIPInputStream(extDataLoader.readData(DataTypes.hotelsExport))
    val lines = IO.readLines(is).flatMap { line =>
      val parts = Tabbed.unapplySeq(line).get

      parts.ensuring(_.length == Fields.values.size,
        s"Invalid line. Columns number doesn't match. Expected: ${Fields.values.size}. " +
          s"Found: ${parts.size}. Line is:\n'$line'\n")

      val fieldWithValue = Fields.values.zip(parts)
      if (fieldWithValue.exists { case (field, value) => !fits(field, value) }) {
        None
      } else {
        val sortedFields: Seq[Field] = fields.toSeq.sortBy(_.id)
        Some(sortedFields.map(field => parts(field.id)).mkString("\t"))
      }
    }
    new StringIteratorInputStream(lines, Some(is))
  }
}
