package ru.yandex.tours.hotels

import scala.collection.mutable
import java.io.InputStream

import org.apache.commons.codec.binary.Base64
import ru.yandex.extdata.common.meta.DataType
import ru.yandex.tours.extdata.{DataDefWithDependencies, DataTypes}
import ru.yandex.tours.model.BaseModel.ProtoImage
import ru.yandex.tours.model.hotels.Hotel
import ru.yandex.tours.model.{Image, Panorama}
import ru.yandex.tours.util.Collections._
import ru.yandex.tours.util.{IO, Logging}
import ru.yandex.tours.util.parsing.{IntValue, Tabbed}
import shapeless.{::, HNil}

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 28.10.16
 */
class HotelsPanoramas(map: Map[Int, Panorama]) {
  def all: Seq[(Int, Panorama)] = map.toSeq
  def getPanorama(hotelId: Int): Option[Panorama] = map.get(hotelId)
  def getPanorama(hotel: Hotel): Option[Panorama] = map.get(hotel.id)
}

object HotelsPanoramas extends DataDefWithDependencies[HotelsPanoramas, HotelsIndex :: HNil] with Logging {
  override def dataType: DataType = DataTypes.hotelsPanoramas
  override def dependsOn: Set[DataType] = DataTypes.shardedHotels.dataTypes.toSet

  override def parse(is: InputStream, dependencies: HotelsIndex :: HNil): HotelsPanoramas = {
    val hotelsIndex = dependencies.head
    var data = mutable.HashMap[Int, Panorama]()
    for (Tabbed(IntValue(hotelId), preview, video) <- IO.readLines(is)) {
      val hotel = hotelsIndex.getHotelById(hotelId)
      if (hotel.isDefined) {
        val image = Image.fromProto(ProtoImage.parseFrom(Base64.decodeBase64(preview)))
        data += hotel.get.id -> Panorama(image, video)
      } else {
        log.error(s"${dataType.getName}: Unknown Hotel Id $hotelId")
      }
    }
    new HotelsPanoramas(data.toMap)
  }
}