package ru.yandex.tours.hotels

import java.io.InputStream

import ru.yandex.extdata.common.meta.DataType
import ru.yandex.tours.extdata.{DataDefWithDependencies, DataTypes}
import ru.yandex.tours.model.hotels.Hotel
import ru.yandex.tours.util.parsing.{DoubleValue, IntValue, Tabbed}
import shapeless._

import scala.collection.mutable
import scala.collection.mutable.ListBuffer
import scala.io.Source

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 29.06.15
 */
class HotelsSimilarity(map: Map[Int, Map[Int, Double]], hotels: HotelsIndex) {
  def getSimilar(hotelId: Int): List[Hotel] = {
    val hotelIds = map.getOrElse(hotelId, Map.empty).toList.sortBy(-_._2).map(_._1)
    hotels.getHotelsById(hotelIds).values.toList
  }

  def getSimilarity(hotelId1: Int, hotelId2: Int): Double = {
    map.getOrElse(hotelId1, Map.empty).getOrElse(hotelId2, 0d)
  }
}

object HotelsSimilarity extends DataDefWithDependencies[HotelsSimilarity, HotelsIndex :: HNil] {
  override def dataType: DataType = DataTypes.hotelsSimilarity
  override def dependsOn: Set[DataType] = DataTypes.shardedHotels.dataTypes.toSet

  override def parse(is: InputStream, dependencies: HotelsIndex :: HNil): HotelsSimilarity = {
    val map = new mutable.HashMap[Int, ListBuffer[(Int, Double)]]
    Source.fromInputStream(is).getLines().foreach {
      case Tabbed(IntValue(hotel1), IntValue(hotel2), DoubleValue(similarity)) =>
        map.getOrElseUpdate(hotel1, ListBuffer.empty) += (hotel2 -> similarity)
        map.getOrElseUpdate(hotel2, ListBuffer.empty) += (hotel1 -> similarity)
    }

    new HotelsSimilarity(map.mapValues(_.toMap).toMap, dependencies.head)
  }
}
