package ru.yandex.tours.hotels

import ru.yandex.tours.geo.distanceInKm
import ru.yandex.tours.model.BaseModel.Point
import ru.yandex.tours.model.hotels.HotelsHolder
import ru.yandex.tours.model.hotels.HotelsHolder._

import scala.collection.JavaConverters._

/* @author berkut@yandex-team.ru */

object HotelsUtil {

  def distance(x: HotelsHolder.Hotel.Builder, y: HotelsHolder.Hotel.Builder): Double = {
    distanceInKm(x.getPoint.getLatitude, x.getPoint.getLongitude, y.getPoint.getLatitude, y.getPoint.getLongitude)
  }

  def distance(x: HotelsHolder.Hotel, y: HotelsHolder.Hotel): Double = {
    distanceInKm(x.getPoint.getLatitude, x.getPoint.getLongitude, y.getPoint.getLatitude, y.getPoint.getLongitude)
  }

  def distance(x: Point, y: Point): Double = {
    distanceInKm(x, y)
  }

  def updateHotels(hotels: TraversableOnce[Hotel], update: Hotel.Builder => Unit): TraversableOnce[Hotel] = {
    hotels.map { parentHotel =>
      val childrenBuilder = parentHotel.getChildrenList.asScala.map(_.toBuilder)
      childrenBuilder.foreach(update)
      val children = childrenBuilder.map(_.build())
      val parentBuilder = parentHotel.toBuilder.clearChildren()
      update(parentBuilder)
      parentBuilder.addAllChildren(children.asJava).build
    }
  }

  def updateTravelHotels(hotels: TraversableOnce[TravelHotel],
                         update: TravelHotel.Builder => Unit): TraversableOnce[TravelHotel] = {
    hotels.map { parentHotel =>
      val parentBuilder = parentHotel.toBuilder
      update(parentBuilder)
      parentBuilder.build
    }
  }

  def convert(hotel: Hotel): TravelHotel = {
    val all = hotel :: hotel.getChildrenList.asScala.toList
    val features = all.flatMap(_.getFeaturesList.asScala)
    val partnerInfo = all.map { x =>
      PartnerInfo.newBuilder()
        .setId(x.getId)
        .setPartner(x.getPartnerId)
        .setPartnerUrl(x.getPartnerUrl)
        .setPartnerId(x.getLocalId)
        .build
    }

    val builder = TravelHotel.newBuilder()
      .setId(hotel.getId)
      .addAllName(hotel.getNameList)
      .addAllAddress(hotel.getAddressList)
      .addAllFeatures(features.asJava)
      .addAllPartnerInfo(partnerInfo.asJava)
      .setStars(hotel.getStars)
      .setGeoId(hotel.getGeoId)
      .addAllPhone(hotel.getPhoneList)
      .addAllSynonyms(hotel.getSynonymsList)
      .addAllImages(hotel.getImagesList)

    if (hotel.hasPoint) builder.setPoint(hotel.getPoint)
    if (hotel.hasReviewsCount) builder.setReviewsCount(hotel.getReviewsCount)
    if (hotel.hasRating) builder.setRating(hotel.getRating)
    if (hotel.hasType) builder.setType(hotel.getType)
    if (hotel.hasHotelUrl) builder.setHotelUrl(hotel.getHotelUrl)

    builder.build
  }

}
