package ru.yandex.tours.hotels

import java.io.{File, FileInputStream}

import ru.yandex.tours.filter.Filters
import ru.yandex.tours.geo.GeoIndex
import ru.yandex.tours.geo.base.region.Tree
import ru.yandex.tours.hotels.MemoryHotelsIndex.TOP
import ru.yandex.tours.model.MapRectangle
import ru.yandex.tours.model.filter.HotelFilter
import ru.yandex.tours.model.geo.MapObject
import ru.yandex.tours.model.hotels.Hotel
import ru.yandex.tours.model.hotels.HotelsHolder.TravelHotel
import ru.yandex.tours.model.hotels.Partners.Partner
import ru.yandex.tours.model.util.Paging
import ru.yandex.tours.util.collections.Bag
import ru.yandex.tours.util.{Logging, ProtoIO}
import shapeless.{::, HNil}

/* @author berkut@yandex-team.ru */

class MemoryHotelsIndex(hotelsForIndex: Iterable[Hotel], tree: Tree, hotelRatings: HotelRatings)
  extends GeoIndex(hotelsForIndex) with HotelsIndex {

  private val hotelsIndex = {
    hotelsForIndex.flatMap(hotel => hotel.partnerIds.map(info => (info.partner, info.id) -> hotel)).toMap
  }
  private val id2hotel = hotelsForIndex.map(hotel => hotel.id -> hotel).toMap
  private val geoId2top = hotelsForIndex.flatMap { hotel =>
    tree.pathToRoot(hotel.geoId).map(_ -> hotel)
  }.groupBy(_._1).map {
    case (region, hotels) => region.id -> hotels.map(_._2).toSeq.sorted(Hotel.ordering.reverse).take(TOP)
  }

  private val region2hotelCount = {
    val result = new Bag[Int]()
    for {
      hotel <- hotelsForIndex
      region <- tree.pathToRoot(hotel.geoId)
    } {
      result += region.id
    }
    result
  }

  private def injectRatings(hotel: Hotel) = {
    hotel.copy(
      rating = hotelRatings.getRating(hotel.id),
      reviewsCount = hotelRatings.getReviewsCount(hotel.id)
    )
  }

  private def fits(filters: Iterable[HotelFilter]): (Hotel) => Boolean = {
    val holdersMap = Filters.hotelHolders.map(h => h.name -> h).toMap
    hotel: Hotel => {
      filters.forall { filter =>
        val values = holdersMap.get(filter.name).map(_.getValues(hotel)).getOrElse(Seq.empty)
        filter.fits(values)
      }
    }
  }

  override def getHotel(partner: Partner, id: String): Option[Hotel] =
    hotelsIndex.get((partner, id)).map(injectRatings)

  override def getHotelById(id: Int): Option[Hotel] = id2hotel.get(id).map(injectRatings)

  override def hotels: Iterator[Hotel] = hotelsForIndex.toIterator.map(injectRatings)

  override def size: Int = hotelsForIndex.size

  override def inRectangle(mapInfo: MapRectangle, maxSize: Int, filters: HotelFilter*): Seq[Hotel] =
    this.inRectangle(mapInfo).filter(fits(filters)).take(maxSize).map(injectRatings)

  override def near(hotel: MapObject, maxSize: Int, filters: HotelFilter*): Iterator[Hotel] =
    this.near(hotel).filter(_ != hotel).filter(fits(filters)).take(maxSize).map(injectRatings)

  override def getHotelsById(ids: Iterable[Int]): Map[Int, Hotel] =
    ids.flatMap(id => getHotelById(id).map(id -> _)).toMap

  override def getHotels(ids: Iterable[Int],
                         filters: Iterable[HotelFilter],
                         mr: Option[MapRectangle]): Map[Int, Hotel] = {
    val predicate = fits(filters)
    getHotelsById(ids).filter {
      case (id, hotel) => predicate(hotel) && mr.forall(_.contains(hotel))
    }
  }

  override def filter(ids: Iterable[Int], filters: Iterable[HotelFilter], mr: Option[MapRectangle]): Set[Int] =
    getHotels(ids, filters, mr).keySet

  override def topInRegion(geoId: Int, maxSize: Int, filters: HotelFilter*): Iterator[Hotel] =
    geoId2top.getOrElse(geoId, Iterable.empty).map(injectRatings).toIterator

  override def hotelsCountInRegion(geoId: Int): Int = region2hotelCount.getCount(geoId)

  override def getHotels(page: Paging, filters: HotelFilter*): Seq[Hotel] = {
    val predicate = fits(filters)
    val hotels = hotelsForIndex.filter(predicate).toSeq
    page(hotels).map(injectRatings)
  }

  override def getHotelsSorted(page: Paging, filters: HotelFilter*): Seq[Hotel] = {
    val predicate = fits(filters)
    val hotels = hotelsForIndex.filter(predicate).toSeq.sorted
    page(hotels).map(injectRatings)
  }

  override def count(filters: HotelFilter*): Int = {
    val predicate = fits(filters)
    hotelsForIndex.count(predicate)
  }

  override def getHotelsCount(ids: Iterable[Int],
                              filters: Iterable[HotelFilter],
                              mapRectangle: Option[MapRectangle]): Int = {
    getHotels(ids, filters, mapRectangle).size
  }
}

object MemoryHotelsIndex extends Logging {
  private val TOP = 6

  def empty: HotelsIndex = new MemoryHotelsIndex(Seq.empty, Tree.empty, HotelRatings.empty)

  def fromFile(file: File, dependencies: Tree :: HotelRatings :: HNil): MemoryHotelsIndex = {
    val tree :: hotelRatings :: HNil = dependencies
    val is = new FileInputStream(file)
    val hotels = ProtoIO.loadFromStream(is, TravelHotel.PARSER).toVector
    log.info(s"Hotels index: ${hotels.size} hotels parsed")
    val (withCoords, noCoordHotels) = hotels.partition(HotelsIndex.hasCoord)
    log.info(s"Hotels index: ${noCoordHotels.size} hotels ignored due to absent of coordinates")
    val (withImages, withoutImages) = withCoords.partition(_.getImagesCount > 0)
    log.info(s"Hotels index: ${withoutImages.size} hotels ignored due to absent of images")
    require(withImages.nonEmpty, "Hotels index should not be empty")
    new MemoryHotelsIndex(withImages.map(Hotel.apply), tree, hotelRatings)
  }
}