package ru.yandex.tours.hotels

import ru.yandex.tours.backa.BackaPermalinks
import ru.yandex.tours.geo.base.{Region, region}
import ru.yandex.tours.hotels.amendings.HotelAmending
import ru.yandex.tours.hotels.enrichers.{GeoIdByPartnerHotelSetter, RegionWithConfidence}
import ru.yandex.tours.model.BaseModel.{LangToVal, ProtoImage}
import ru.yandex.tours.model.Languages
import ru.yandex.tours.model.hotels.HotelsHolder._
import ru.yandex.tours.model.hotels.Partners
import ru.yandex.tours.model.hotels.Partners._
import ru.yandex.tours.util.Logging

import scala.collection.JavaConverters._
import scala.collection.mutable

class TravelHotelBuilder(allHotels: Iterable[PartnerHotel],
                         ratings: HotelRatings,
                         geoIdSetter: GeoIdByPartnerHotelSetter,
                         tree: region.Tree,
                         amendments: Iterable[HotelAmending],
                         backaPermalinks: BackaPermalinks) extends Logging {

  import TravelHotelBuilder._
  
  val (deleted, hotels) = allHotels.partition(_.getIsDeleted)

  val sorted: Seq[PartnerHotel] = sortCluster(hotels)

  val (hidden, core) = sorted.partition(h => hiddenPartners.contains(h.getRawHotel.getPartner))

  val partnerInfo: Seq[PartnerInfo] = sorted.map { h =>
    val raw = h.getRawHotel
    val b = PartnerInfo.newBuilder()
      .setId(h.getId)
      .setPartner(raw.getPartner)
      .setPartnerUrl(raw.getPartnerUrl)
      .setPartnerId(raw.getPartnerId)
    if (raw.hasRegionId) b.setPartnerRegion(raw.getRegionId)
    b.build()
  }

  def toProto: Option[TravelHotel] = {
    if (core.isEmpty) return None
    val (booking, other) = core.partition(_.getRawHotel.getPartner == Partners.booking.id)
    // If only partner with photos is booking and there are some other partners in cluster we can take only 1 image
    val images = nonEmptyImages(other).orElse {
      nonEmptyImages(booking).map { bookingImages =>
        if (other.nonEmpty) bookingImages.take(1) else bookingImages
      }
    }.getOrElse(List.empty).map(_.toBuilder.clearPHash().clearNNetFeatures().build)

    val phones = core.find(_.getRawHotel.getPhoneCount > 0)
      .map(_.getRawHotel.getPhoneList.asScala).getOrElse(List.empty[String])

    val address = resolveAddress(core.flatMap(_.getRawHotel.getAddressList.asScala))
    val features = core.flatMap(_.getRawHotel.getFeaturesList.asScala)

    val builder = TravelHotel.newBuilder()
      .setId(core.head.getId)
      .addAllAddress(address.asJava)
      .addAllFeatures(features.asJava)
      .addAllPartnerInfo(partnerInfo.asJava)
      .addAllImages(images.asJava)
      .addAllPhone(phones.asJava)

    deleted.map(_.getId).foreach(builder.addDeletedIds)

    core.map(_.getRawHotel).find(_.hasStars).map(_.getStars).foreach(builder.setStars)

    val bestRegion = selectBestRegion(core.flatMap(geoIdSetter.getGeoId))
    bestRegion.foreach(gwc => builder.setGeoId(gwc.region.id))

    val (names, synonyms) = resolveNames(core.map(_.getRawHotel.getNameList.asScala), bestRegion.map(_.region))

    builder
      .addAllName(names.asJava)
      .addAllSynonyms(synonyms.flatten.asJava)

    sorted
      .find(_.getRawHotel.hasHotelUrl)
      .map(_.getRawHotel.getHotelUrl)
      .foreach(builder.setHotelUrl)

    sorted.map(_.getRawHotel).filter(_.hasPoint).map(_.getPoint)
      .find(h => !HotelsIndex.isEmptyPoint(h))
      .foreach(builder.setPoint)

    core.find(_.hasType).map(_.getType).foreach(builder.setType)

    core.map(h => ratings.getRating(h.getId)).find(_ != 0)
      .orElse(core.find(_.getRawHotel.hasRating).map(_.getRawHotel.getRating))
      .foreach(builder.setRating)

    core.map(h => ratings.getReviewsCount(h.getId)).find(_ != 0)
      .foreach(builder.setReviewsCount)

    amendments.toSeq.sortBy(_.timestamp).foreach(_.update(builder, core))

    val permalinks = core.map(h => backaPermalinks.getBackaPermalink(h.getId)).filter(_.nonEmpty)
    if (permalinks.nonEmpty) {
      builder.setBackaPermalink(permalinks.head.get)
    }

    Some(builder.build())
  }

  private def resolveNames(allNames: Seq[Iterable[LangToVal]],
                           region: Option[Region]): (Iterable[LangToVal], Iterable[Seq[LangToVal]]) = {
    val inRKUB = region.flatMap(tree.country).exists(country => RKUB.contains(country.id))

    val plain = allNames.zipWithIndex.flatMap { case (n, index) => n.map(_ -> index) }
    val fixedNames = plain.map { case (name, index) =>
      val fixedName =
        if (!cyrillicLanguages.contains(name.getLang) && isRussian(name.getValue)) {
          name.toBuilder.setLang(Languages.ru.toString).build
        } else {
          name
        }
      fixedName -> index
    }
    val nameMap = fixedNames.groupBy(_._1.getLang).map { case (lang, indexedNames) =>
      lang -> indexedNames.sortBy(_._2).map(_._1)
    }
    nameMap.toVector.map { case (lang, names) =>
      if (lang == Languages.ru.toString && !inRKUB) {
        nameMap.get(Languages.en.toString) match {
          case Some(Seq(enName, _*)) =>
            enName.toBuilder.setLang(Languages.ru.toString).build() -> names
          case _ =>
            names.head -> names.tail
        }
      } else {
        names.head -> names.tail
      }
    }.unzip
  }

  private def nonEmptyImages(hotels: Seq[PartnerHotel]): Option[Seq[ProtoImage]] = {
    val sorted = sortClusterByImages(hotels)
    sorted.find { h =>
      (!imagelessPartners.contains(h.getRawHotel.getPartner)) && (h.getImagesCount > 0)
    }.map(_.getImagesList.asScala)
  }
}

object TravelHotelBuilder {
  private val RKUB = Set(225, 159, 187, 149) // Russia, Kazakhstan, Ukraine, Belarus

  def sortCluster(hotels: Iterable[PartnerHotel]): Seq[PartnerHotel] = {
    hotels.toSeq.sortBy(h => (partnerPriority.getOrElse(h.getRawHotel.getPartner, 100), h.getId))
  }

  def sortClusterByImages(hotels: Iterable[PartnerHotel]): Seq[PartnerHotel] = {
    hotels.toSeq.sortBy(h => (imagesPriority.getOrElse(h.getRawHotel.getPartner, 100), h.getId))
  }

  def buildTravelHotel(hotels: Iterable[PartnerHotel],
                       ratings: HotelRatings,
                       geoIdSetter: GeoIdByPartnerHotelSetter,
                       tree: region.Tree,
                       amendments: Iterable[HotelAmending],
                       backaPermalinks: BackaPermalinks): Option[TravelHotel] = {
    new TravelHotelBuilder(hotels, ratings, geoIdSetter, tree, amendments, backaPermalinks).toProto
  }

  private val hiddenPartners = Set(Partners.backa, Partners.topHotels, Partners.inna, Partners.ostrovok, Partners.oktogo).map(_.id)
  private val imagelessPartners = Set(Partners.backa).map(_.id)

  private val russianLetters = "йцукенгшщзхъфывапролджэёячсмитьбю".toSet

  private val cyrillicLanguages = Set(Languages.ru, Languages.by, Languages.tt, Languages.ua, Languages.uk)
    .map(_.toString)

  private def isRussian(x: String): Boolean = {
    x.exists(russianLetters.contains)
  }

  private def selectBestRegion(candidates: Seq[RegionWithConfidence]): Option[RegionWithConfidence] = {
    candidates.sortBy(-_.confidence).headOption.flatMap { case RegionWithConfidence(_, conf) =>
      val withBBox = candidates.filter(_.confidence >= conf - 1e-6)
        .filter(_.region.boundingBox.nonEmpty)
      if (withBBox.nonEmpty) {
        Some(withBBox.minBy(_.region.boundingBox.area))
      } else {
        candidates.headOption
      }
    }
  }

  private def resolveAddress(addresses: Seq[Address]): Iterable[Address] = {
    val result = mutable.Map.empty[String, Address.Builder]
    for (address <- addresses) {
      val lang = address.getLang
      val toUpdate = result.getOrElseUpdate(lang, address.toBuilder)
      if (!toUpdate.hasCountry && address.hasCountry) toUpdate.setCountry(address.getCountry)
      if (!toUpdate.hasAdminName && address.hasAdminName) toUpdate.setAdminName(address.getAdminName)
      if (!toUpdate.hasLocality && address.hasLocality) toUpdate.setLocality(address.getLocality)
      if (!toUpdate.hasSubAdminName && address.hasSubAdminName) toUpdate.setSubAdminName(address.getSubAdminName)
      if (!toUpdate.hasStreet && address.hasStreet) toUpdate.setStreet(address.getStreet)
      if (!toUpdate.hasHouse && address.hasHouse) toUpdate.setHouse(address.getHouse)
      if (!toUpdate.hasFullAddress && address.hasFullAddress) toUpdate.setFullAddress(address.getFullAddress)
    }
    result.values.map(_.build())
  }

  private val partnerPriority = Map(
    lt -> 0,
    sodis -> 1,
    hotels101 -> 2,
    ostrovok -> 3,
    ostrovokv3 -> 4,
    booking -> 5,
    hotelsCombined -> 7
  ).map {
    case (partner, priority) => partner.id -> priority
  }

  private val imagesPriority = Map(
    ostrovokv3 -> 1,
    sodis -> 2,
    hotels101 -> 3,
    ostrovok -> 4,
    booking -> 5,
    hotelsCombined -> 6,
    lt -> 7
  ).map {
    case (partner, priority) => partner.id -> priority
  }
}
