package ru.yandex.tours.hotels

import java.io._

import com.yandex.yoctodb.DatabaseFormat
import com.yandex.yoctodb.immutable.Database
import com.yandex.yoctodb.mutable.DocumentBuilder.IndexOption.{FILTERABLE, SORTABLE}
import com.yandex.yoctodb.query.QueryBuilder._
import com.yandex.yoctodb.query._
import com.yandex.yoctodb.query.simple.SimpleDescendingOrder
import com.yandex.yoctodb.util.UnsignedByteArrays
import com.yandex.yoctodb.util.buf.{Buffer, BufferInputStream}
import ru.yandex.tours.filter._
import ru.yandex.tours.geo
import ru.yandex.tours.geo.base.region.Tree
import ru.yandex.tours.model.MapRectangle
import ru.yandex.tours.model.filter._
import ru.yandex.tours.model.filter.hotel.{GeoIdFilter, PartnerIdFilter}
import ru.yandex.tours.model.geo.MapObject
import ru.yandex.tours.model.hotels.Hotel
import ru.yandex.tours.model.hotels.HotelsHolder.TravelHotel
import ru.yandex.tours.model.hotels.Partners.Partner
import ru.yandex.tours.model.search.SearchType
import ru.yandex.tours.model.util.Paging
import ru.yandex.tours.util.Logging
import ru.yandex.tours.yocto.DatabaseWrapper

import scala.collection.JavaConverters._
import scala.collection.mutable

object YoctoHotelsIndex extends Logging {

  object Fields {
    val ID = "id"
    val PARTNER_LOCAL_ID = "p_id"
    val PARTNER_ID = PartnerIdFilter.name
    val LONGITUDE = "lon"
    val LATITUDE = "lat"
    val RELEVANCE = "rel"
    val GEO_ID = GeoIdFilter.name
    val SEARCH_TYPE = "st"
  }

  def readProtoHotel(file: File): Iterator[TravelHotel] = {
    val db = DatabaseFormat.getCurrent.getDatabaseReader.from(Buffer.mmap(file, false))
    (0 until db.getDocumentCount).toIterator.map(i => TravelHotel.parseFrom(db.getDocument(i).toByteArray))
  }

  private def getPartnerLocalId(partnerId: Int, localId: String): String = partnerId + "_" + localId

  private def getDoubleIndex(x: Double) = (x * 1000000).toInt

  def build(hotels: TraversableOnce[TravelHotel], tree: Tree, hotelRatings: HotelRatings, os: OutputStream) {
    val dbBuilder = DatabaseFormat.getCurrent.newDatabaseBuilder
    var okHotels = 0
    var ignored = 0
    hotels.foreach { hotel =>
      if (!HotelsIndex.isIndexable(hotel)) {
        ignored += 1
      } else {
        val modelHotel = Hotel(hotel)
        okHotels += 1
        var relevance = hotelRatings.getRelevance(modelHotel.id)
        if (relevance == 0) relevance = modelHotel.relevance
        var builder = DatabaseFormat.getCurrent.newDocumentBuilder()
          .withField(Fields.LONGITUDE, getDoubleIndex(hotel.getPoint.getLongitude), FILTERABLE)
          .withField(Fields.LATITUDE, getDoubleIndex(hotel.getPoint.getLatitude), FILTERABLE)
          .withField(Fields.RELEVANCE, relevance.toInt, SORTABLE)
          .withPayload(hotel.toByteArray)

        for (filter <- Filters.hotelHolders) {
          filter.getValues(modelHotel).foreach {
            case StringValue(value) => builder = builder.withField(filter.name, value, FILTERABLE)
            case BooleanValue(value) => builder = builder.withField(filter.name, value, FILTERABLE)
            case IntValue(value) => builder = builder.withField(filter.name, value, FILTERABLE)
            case DoubleValue(value) => builder = builder.withField(filter.name, getDoubleIndex(value), FILTERABLE)
            case NullValue =>
          }
        }

        tree.findParents(hotel.getGeoId).foreach { region =>
          builder = builder.withField(Fields.GEO_ID, region.id, FILTERABLE)
        }

        if (modelHotel.roomsSearchAvailable) {
          builder = builder.withField(Fields.SEARCH_TYPE, SearchType.ROOMS.id, FILTERABLE)
        }

        if (modelHotel.toursSearchAvailable) {
          builder = builder.withField(Fields.SEARCH_TYPE, SearchType.TOURS.id, FILTERABLE)
        }
        hotel.getPartnerInfoList.asScala.foreach { pi =>
          builder = builder
            .withField(Fields.PARTNER_LOCAL_ID, getPartnerLocalId(pi.getPartner, pi.getPartnerId), FILTERABLE)
            .withField(Fields.PARTNER_ID, pi.getPartner, FILTERABLE)
            .withField(Fields.ID, pi.getId, FILTERABLE)
        }
        hotel.getDeletedIdsList.asScala.foreach { deletedId =>
          builder = builder.withField(Fields.ID, deletedId, FILTERABLE)
        }

        dbBuilder.merge(builder)
      }
    }
    dbBuilder.buildWritable().writeTo(os)
    log.info(s"Yocto hotels index built. $okHotels hotels in index. $ignored hotels ignored")
  }

  // only for tests
  def fromFile(file: File, ratings: HotelRatings = HotelRatings.empty): YoctoHotelsIndex = {
    val db = DatabaseFormat.getCurrent.getDatabaseReader.from(Buffer.mmap(file, false))
    new YoctoHotelsIndex(ratings, DatabaseWrapper(db))
  }

  // only for testing
  def fromBytes(ratings: HotelRatings, bytes: Array[Byte]): YoctoHotelsIndex = {
    val buffer = Buffer.from(bytes)
    val db = DatabaseFormat.getCurrent.getDatabaseReader.from(buffer)
    new YoctoHotelsIndex(HotelRatings.empty, DatabaseWrapper(db))
  }
}

class YoctoHotelsIndex(hotelRatings: HotelRatings, wrappers: DatabaseWrapper*) extends HotelsIndex with Logging {

  private def db = DatabaseFormat.getCurrent.getDatabaseReader.composite(wrappers.map(_.db).asJava)

  import YoctoHotelsIndex._

  private def parseId(buffer: Buffer): Int = {
    TravelHotel.parseFrom(new BufferInputStream(buffer)).getId
  }

  private def parseHotel(buffer: Buffer): Hotel = {
    val hotel = Hotel(TravelHotel.parseFrom(new BufferInputStream(buffer)))
    hotel.copy(rating = hotelRatings.getRating(hotel.id), reviewsCount = hotelRatings.getReviewsCount(hotel.id))
  }

  override def getHotel(partner: Partner, id: String): Option[Hotel] = {
    val condition = QueryBuilder.eq(Fields.PARTNER_LOCAL_ID, UnsignedByteArrays.from(getPartnerLocalId(partner.id, id)))
    execute(select().where(condition), maxSize = 1).headOption
  }

  override def getHotelById(id: Int): Option[Hotel] = {
    val query = select().where(QueryBuilder.eq(Fields.ID, UnsignedByteArrays.from(id)))
    execute(query, maxSize = 1).headOption
  }

  override def inRectangle(mapInfo: MapRectangle, maxSize: Int, filters: HotelFilter*): Seq[Hotel] = {
    val query = queryFilters(select, filters, Some(mapInfo))
    execute(query, maxSize)
  }

  def hotels: Iterator[Hotel] = {
    val currentDb = db
    (0 until currentDb.getDocumentCount).toIterator.map(i => parseHotel(currentDb.getDocument(i)))
  }

  override def size: Int = db.getDocumentCount

  override def near(obj: MapObject, maxSize: Int, filters: HotelFilter*): Iterator[Hotel] = {
    var found = Set.empty[Hotel]
    Iterator.iterate(0.1)(_ * 2).takeWhile(_ => found.size < maxSize).foreach { step =>
      val rect = MapRectangle.byCenterAndSpan(obj.longitude, obj.latitude, step, step)
      val chunk = inRectangle(rect, maxSize * 2, filters: _*)
        .filter(_ != obj)
        .filterNot(found.contains)
      found ++= chunk
    }
    found.toSeq.sortBy(geo.distanceInKm(_, obj))
      .take(maxSize)
      .iterator
  }

  override def getHotelsById(ids: Iterable[Int]): Map[Int, Hotel] = {
    if (ids.isEmpty) {
      Map.empty
    } else {
      val query = select().where(in(Fields.ID, ids.toSeq.distinct.map(UnsignedByteArrays.from): _*))
      execute(query, Integer.MAX_VALUE).map(h => h.id -> h).toMap
    }
  }

  override def getHotels(ids: Iterable[Int], filters: Iterable[HotelFilter],
                         mapRectangle: Option[MapRectangle]): Map[Int, Hotel] = {
    if (ids.isEmpty) {
      Map.empty
    } else {
      val query = queryForHotelsWithFilters(ids, filters, mapRectangle)
      execute(query, Integer.MAX_VALUE).map(h => h.id -> h).toMap
    }
  }

  override def getHotelsCount(ids: Iterable[Int],
                              filters: Iterable[HotelFilter],
                              mapRectangle: Option[MapRectangle]): Int = {
    if (ids.isEmpty) return 0
    val query = queryForHotelsWithFilters(ids, filters, mapRectangle)
    db.count(query)
  }

  override def filter(ids: Iterable[Int], filters: Iterable[HotelFilter],
                      mapRectangle: Option[MapRectangle]): Set[Int] = {
    if (ids.isEmpty) {
      Set.empty
    } else {
      val query = queryForHotelsWithFilters(ids, filters, mapRectangle)
      executeIds(query, Integer.MAX_VALUE).toSet
    }
  }

  override def topInRegion(geoId: Int, maxSize: Int, filters: HotelFilter*): Iterator[Hotel] = {
    val selectQuery = queryFilters(select(), filters, None)
      .where(QueryBuilder.eq(Fields.GEO_ID, UnsignedByteArrays.from(geoId)))
    val resultQuery = selectQuery.orderBy(new SimpleDescendingOrder(Fields.RELEVANCE))
    execute(resultQuery, maxSize).iterator
  }

  override def hotelsCountInRegion(geoId: Int): Int = {
    val query = select.where(QueryBuilder.eq(Fields.GEO_ID, UnsignedByteArrays.from(geoId)))
    db.count(query)
  }

  override def getHotels(page: Paging, filters: HotelFilter*): Seq[Hotel] = {
    val query = queryFilters(select(), filters, None).skip(page.page * page.pageSize).limit(page.pageSize)
    execute(query, Int.MaxValue)
  }

  override def getHotelsSorted(page: Paging, filters: HotelFilter*): Seq[Hotel] = {
    val query = queryFilters(select(), filters, None)
      .orderBy(new SimpleDescendingOrder(Fields.RELEVANCE))
      .skip(page.page * page.pageSize)
      .limit(page.pageSize)
    execute(query, Int.MaxValue)
  }

  override def count(filters: HotelFilter*): Int = {
    val query = queryFilters(select(), filters, None)
    db.count(query)
  }

  private def queryFilters(query: Select, filters: Iterable[HotelFilter], optMapRectangle: Option[MapRectangle]) = {
    var q = query
    for (filter <- filters) {
      val values = filter.values.collect {
        case StringValue(value) => UnsignedByteArrays.from(value)
        case IntValue(value) => UnsignedByteArrays.from(value)
        case BooleanValue(value) => UnsignedByteArrays.from(value)
      }
      if (values.nonEmpty) {
        q = q.where(in(filter.name, values: _*))
      }
    }
    optMapRectangle.foreach { mr =>
      q = q.where(
        in(Fields.LATITUDE,
          UnsignedByteArrays.from(getDoubleIndex(mr.minLat)), true,
          UnsignedByteArrays.from(getDoubleIndex(mr.maxLat)), true
        )
      )
      q = if (mr.minLon < mr.maxLon) {
        q.where(
          in(Fields.LONGITUDE,
            UnsignedByteArrays.from(getDoubleIndex(mr.minLon)), true,
            UnsignedByteArrays.from(getDoubleIndex(mr.maxLon)), true
          )
        )
      } else {
        q.where(
          or(
            lte(Fields.LONGITUDE, UnsignedByteArrays.from(getDoubleIndex(mr.maxLon))),
            gte(Fields.LONGITUDE, UnsignedByteArrays.from(getDoubleIndex(mr.minLon)))
          )
        )
      }
    }
    q
  }

  private def queryForHotelsWithFilters(ids: Iterable[Int], filters: Iterable[HotelFilter],
                                        mapRectangle: Option[MapRectangle]) = {
    val query = select().where(in(Fields.ID, ids.toSeq.distinct.map(UnsignedByteArrays.from): _*))
    queryFilters(query, filters, mapRectangle)
  }

  private def execute(q: Query, maxSize: Int): Seq[Hotel] = {
    executeQuery(q, maxSize)(parseHotel)
  }

  private def executeIds(q: Query, maxSize: Int): Seq[Int] = {
    executeQuery(q, maxSize)(parseId)
  }

  private def executeQuery[T](q: Query, maxSize: Int)(parser: Buffer => T): Seq[T] = {
    val debug = log.isDebugEnabled
    val start = if (debug) System.nanoTime() else 0

    val result = mutable.Buffer.empty[T]
    var collected = 0
    if (maxSize > 0) {
      db.execute(q, new DocumentProcessor {
        override def process(document: Int, database: Database): Boolean = {
          result += parser(database.getDocument(document))
          collected += 1
          collected < maxSize
        }
      })
    }
    if (debug) {
      log.debug(s"Executed $q in ${(System.nanoTime() - start) / 1000000} ms.")
    }
    result
  }
}
