package ru.yandex.tours.hotels.amendings

import java.nio.ByteBuffer

import com.google.common.base.Charsets
import com.google.protobuf.ByteString
import ru.yandex.tours.model.BaseModel.{Point, ProtoImage}
import ru.yandex.tours.model.Languages.Lang
import ru.yandex.tours.model.hotels.HotelsHolder.TravelHotel.Builder
import ru.yandex.tours.model.hotels.HotelsHolder._
import ru.yandex.tours.util.parsing.{IntValue, Tabbed}
import HotelAmending.Charset
import ru.yandex.tours.model.Languages

import scala.collection.JavaConverters._

trait HotelAmending {
  def update(builder: TravelHotel.Builder, hotels: Seq[PartnerHotel]): Unit

  def timestamp: Long

  def hotelId: Int
  
  def serialize: Array[Byte]

  def `type`: AmendingType

  def toProto: ProtoHotelAmending = {
    ProtoHotelAmending.newBuilder()
      .setTimestamp(timestamp)
      .setHotelId(hotelId)
      .setPayload(ByteString.copyFrom(serialize))
      .setType(`type`.id)
      .build()
  }
}

object HotelAmending {
  private[amendings] val Charset = Charsets.UTF_8

  def parseProto(proto: ProtoHotelAmending): HotelAmending = {
    parse(proto.getType, proto.getHotelId, proto.getTimestamp, proto.getPayload.toByteArray)
  }

  def parseProtos(proto: ProtoHotelAmendments): Seq[HotelAmending] = {
    proto.getAmendingsList.asScala.map(parseProto)
  }

  def parse(typeId: Int, hotelId: Int, timestamp: Long, payload: Array[Byte]): HotelAmending = {
    val `type` = AmendingType.byId(typeId)
    `type` match {
      case AmendingType.SET_GEO_ID =>
        val geoId = ByteBuffer.wrap(payload).getInt
        SetGeoIdAmending(hotelId, timestamp, geoId)
      case AmendingType.SET_POINT =>
        val point = Point.parseFrom(payload)
        SetPointAmending(hotelId, timestamp, point)
      case AmendingType.SELECT_POINT =>
        val IntValue(partnerId) = new String(payload, Charset)
        SelectPointAmending(hotelId, timestamp, partnerId)
      case AmendingType.ADD_FEATURE =>
        val parts = new String(payload, Charset).split("\t")
        AddFeatureAmending(hotelId, timestamp, parts.head.intern(), parts.tail: _*)
      case AmendingType.DELETE_FEATURE =>
        val name = new String(payload, Charset)
        DeleteFeatureAmending(hotelId, timestamp, name.intern())
      case AmendingType.SELECT_FEATURE =>
        val Array(name, IntValue(partnerId)) = new String(payload, Charset).split("\t")
        SelectFeatureAmending(hotelId, timestamp, name.intern(), partnerId)
      case AmendingType.ADD_IMAGE =>
        val image = ProtoImage.parseFrom(payload)
        AddImage(hotelId, timestamp, image)
      case AmendingType.DELETE_IMAGE =>
        val name = new String(payload, Charset)
        DeleteImageAmending(hotelId, timestamp, name)
      case AmendingType.SET_FIRST_IMAGE =>
        val name = new String(payload, Charset)
        SetFirstImageAmending(hotelId, timestamp, name)
      case AmendingType.SET_HOTEL_TYPE =>
        val id = ByteBuffer.wrap(payload).getInt
        val newType = HotelType.valueOf(id)
        SetHotelTypeAmending(hotelId, timestamp, newType)
      case AmendingType.SET_STARS =>
        val IntValue(value) = new String(payload, Charset)
        SetStarsAmending(hotelId, timestamp, value)
      case AmendingType.SELECT_STARS =>
        val IntValue(partnerId) = new String(payload, Charset)
        SelectStarsAmending(hotelId, timestamp, partnerId)
      case AmendingType.SELECT_ADDRESS =>
        val IntValue(partnerId) = new String(payload, Charset)
        SelectAddressAmending(hotelId, timestamp, partnerId)
      case AmendingType.SELECT_HOTEL_TYPE =>
        val IntValue(partnerHotelId) = new String(payload, Charset)
        SelectHotelTypeAmending(hotelId, timestamp, partnerHotelId)
      case AmendingType.SET_HOTEL_NAME ⇒
        val Tabbed(lang, name) = new String(payload, Charset)
        SetHotelName(hotelId, timestamp, Languages.withName(lang), name)
      case AmendingType.DELETE_HOTEL_NAME ⇒
        val lang = new String(payload, Charset)
        DeleteHotelName(hotelId, timestamp, Languages.withName(lang))
    }
  }
}

case class SetGeoIdAmending(hotelId: Int, timestamp: Long, geoId: Int) extends HotelAmending {
  override def update(builder: TravelHotel.Builder, hotels: Seq[PartnerHotel]): Unit = builder.setGeoId(geoId)

  override def serialize: Array[Byte] = ByteBuffer.allocate(4).putInt(geoId).array()

  override def `type`: AmendingType = AmendingType.SET_GEO_ID
}

case class SetHotelTypeAmending(hotelId: Int, timestamp: Long, newType: HotelType) extends HotelAmending {
  override def update(builder: TravelHotel.Builder, hotels: Seq[PartnerHotel]): Unit = builder.setType(newType)

  override def serialize: Array[Byte] = ByteBuffer.allocate(4).putInt(newType.getNumber).array()

  override def `type`: AmendingType = AmendingType.SET_HOTEL_TYPE
}

case class SelectHotelTypeAmending(hotelId: Int, timestamp: Long, partnerHotelId: Int) extends HotelAmending {
  override def update(builder: TravelHotel.Builder, hotels: Seq[PartnerHotel]): Unit = {
    val tpe = hotels.find(_.getId == partnerHotelId).filter(_.hasType).map(_.getType)
    tpe.foreach(builder.setType)
  }

  override def serialize: Array[Byte] = partnerHotelId.toString.getBytes(Charset)

  override def `type`: AmendingType = AmendingType.SELECT_HOTEL_TYPE
}

case class SetPointAmending(hotelId: Int, timestamp: Long, point: Point) extends HotelAmending {
  override def update(builder: TravelHotel.Builder, hotels: Seq[PartnerHotel]): Unit = builder.setPoint(point)

  override def serialize: Array[Byte] = point.toByteArray

  override def `type`: AmendingType = AmendingType.SET_POINT
}

case class SelectPointAmending(hotelId: Int, timestamp: Long, partnerHotelId: Int) extends HotelAmending {
  override def update(builder: TravelHotel.Builder, hotels: Seq[PartnerHotel]): Unit = {
    val point = hotels.find(_.getId == partnerHotelId).map(_.getRawHotel).filter(_.hasPoint).map(_.getPoint)
    point.foreach(builder.setPoint)
  }

  override def serialize: Array[Byte] = partnerHotelId.toString.getBytes(Charset)

  override def `type`: AmendingType = AmendingType.SELECT_POINT
}

case class AddFeatureAmending(hotelId: Int, timestamp: Long, name: String, values: String*) extends HotelAmending {
  override def update(builder: TravelHotel.Builder, hotels: Seq[PartnerHotel]): Unit = {
    val features = builder.getFeaturesList.asScala.filterNot(_.getName == name)
    val newFeatures = values.map(Feature.newBuilder().setName(name).setValue(_).build())
    builder.clearFeatures().addAllFeatures(features.asJava).addAllFeatures(newFeatures.asJava)
  }

  override def serialize: Array[Byte] = (Seq(name) ++ values).mkString("\t").getBytes(Charset)

  override def `type`: AmendingType = AmendingType.ADD_FEATURE
}

case class DeleteFeatureAmending(hotelId: Int, timestamp: Long, name: String) extends HotelAmending {
  override def update(builder: TravelHotel.Builder, hotels: Seq[PartnerHotel]): Unit = {
    val features = builder.getFeaturesList.asScala.filterNot(_.getName == name)
    builder.clearFeatures().addAllFeatures(features.asJava)
  }

  override def serialize: Array[Byte] = name.getBytes(Charset)

  override def `type`: AmendingType = AmendingType.DELETE_FEATURE
}

case class SelectFeatureAmending(hotelId: Int, timestamp: Long,
                                 name: String,
                                 partnerHotelId: Int) extends HotelAmending {

  override def update(builder: TravelHotel.Builder, hotels: Seq[PartnerHotel]): Unit = {
    val features = builder.getFeaturesList.asScala.filterNot(_.getName == name)
    val newFeatures = hotels.find(_.getId == partnerHotelId).toSeq.flatMap {
      hotel => hotel.getRawHotel.getFeaturesList.asScala.filter(_.getName == name)
    }
    builder.clearFeatures().addAllFeatures(features.asJava).addAllFeatures(newFeatures.asJava)
  }

  override def serialize: Array[Byte] = Tabbed(name, partnerHotelId).getBytes(Charset)

  override def `type`: AmendingType = AmendingType.SELECT_FEATURE
}

case class SetStarsAmending(hotelId: Int, timestamp: Long,
                            value: Int) extends HotelAmending {

  override def update(builder: TravelHotel.Builder, hotels: Seq[PartnerHotel]): Unit = {
    builder.setStars(value)
  }

  override def serialize: Array[Byte] = value.toString.getBytes(Charset)

  override def `type`: AmendingType = AmendingType.SET_STARS
}

case class SelectStarsAmending(hotelId: Int, timestamp: Long, partnerHotelId: Int) extends HotelAmending {

  override def update(builder: TravelHotel.Builder, hotels: Seq[PartnerHotel]): Unit = {
    val stars = hotels.find(_.getId == partnerHotelId).map(_.getRawHotel.getStars)
    stars.foreach(s => builder.setStars(s))
  }

  override def serialize: Array[Byte] = partnerHotelId.toString.getBytes(Charset)

  override def `type`: AmendingType = AmendingType.SELECT_STARS
}

case class SelectAddressAmending(hotelId: Int, timestamp: Long, partnerHotelId: Int) extends HotelAmending {

  override def update(builder: TravelHotel.Builder, hotels: Seq[PartnerHotel]): Unit = {
    val address = hotels.find(_.getId == partnerHotelId)
      .filter(_.getRawHotel.getAddressCount > 0)
      .map(_.getRawHotel.getAddressList)
    address.foreach(a => builder.clearAddress.addAllAddress(a))
  }

  override def serialize: Array[Byte] = partnerHotelId.toString.getBytes(Charset)

  override def `type`: AmendingType = AmendingType.SELECT_ADDRESS
}

case class AddImage(hotelId: Int, timestamp: Long, image: ProtoImage) extends HotelAmending {
  override def update(builder: TravelHotel.Builder, hotels: Seq[PartnerHotel]): Unit = {
    builder.addImages(image)
  }

  override def serialize: Array[Byte] = image.toByteArray

  override def `type`: AmendingType = AmendingType.ADD_IMAGE
}

case class DeleteImageAmending(hotelId: Int, timestamp: Long, name: String) extends HotelAmending {
  override def update(builder: TravelHotel.Builder, hotels: Seq[PartnerHotel]): Unit = {
    val images = builder.getImagesList.asScala.filterNot(_.getName == name)
    builder.clearImages().addAllImages(images.asJava)
  }

  override def serialize: Array[Byte] = name.getBytes(Charset)

  override def `type`: AmendingType = AmendingType.DELETE_IMAGE
}

case class SetFirstImageAmending(hotelId: Int, timestamp: Long, name: String) extends HotelAmending {
  override def update(builder: TravelHotel.Builder, hotels: Seq[PartnerHotel]): Unit = {
    val (head, tail) = builder.getImagesList.asScala.partition(_.getName == name)
    builder.clearImages().addAllImages(head.asJava).addAllImages(tail.asJava)
  }

  override def serialize: Array[Byte] = name.getBytes(Charset)

  override def `type`: AmendingType = AmendingType.SET_FIRST_IMAGE
}

case class SetHotelName(hotelId: Int, timestamp: Long, lang: Lang, value: String) extends HotelAmending {
  override def update(builder: Builder, hotels: Seq[PartnerHotel]): Unit = {
    builder.getNameBuilderList.asScala.find(_.getLang == lang.toString) match {
      case Some(v) ⇒ v.setValue(value)
      case None ⇒ builder.addNameBuilder().setLang(lang.toString).setValue(value)
    }
  }

  override def serialize: Array[Byte] = Tabbed(lang, value).getBytes(Charset)

  override def `type`: AmendingType = AmendingType.SET_HOTEL_NAME
}
case class DeleteHotelName(hotelId: Int, timestamp: Long, lang: Lang) extends HotelAmending {
  override def update(builder: Builder, hotels: Seq[PartnerHotel]): Unit = {
    val idx = (0 until builder.getNameCount).find(builder.getName(_).getLang == lang.toString)
    idx.foreach(i ⇒ builder.removeName(i))
  }

  override def serialize: Array[Byte] = lang.toString.getBytes(Charset)

  override def `type`: AmendingType = AmendingType.DELETE_HOTEL_NAME
}