package ru.yandex.tours.index

import java.io.File

import akka.actor.ActorSystem
import org.joda.time.DateTime
import ru.yandex.tours.util.Logging

import scala.concurrent.duration.FiniteDuration
import scala.util.{Failure, Success, Try}

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 12.03.15
 */
object IndexCleaner extends Logging {

  def start(dir: File, extension: String, ttl: FiniteDuration, interval: FiniteDuration)(implicit akka: ActorSystem) = {
    import akka.dispatcher
    akka.scheduler.schedule(interval, interval) {
      IndexCleaner.cleanDirectory(dir, extension, ttl + interval * 2)
    }
  }

  def cleanDirectory(dir: File, extension: String, ttl: FiniteDuration): Unit = {
    require(dir.exists() || dir.mkdirs(), s"[$dir] does not exists")
    log.info(s"Cleaning directory [$dir] ttl = $ttl")
    for (file <- dir.listFiles() if file.getName.endsWith(extension)) {
      Try(file.getName.split("_")(0).toLong) match {
        case Success(creationTime) =>
          if (System.currentTimeMillis() >= creationTime + ttl.toMillis) {
            log.info("Removing old shard " + file + " created at " + new DateTime(creationTime))
            file.delete()
          }
        case Failure(e) =>
          log.info("Removing shard with invalid naming " + file)
          file.delete()
      }
    }
  }
}
