package ru.yandex.tours.operators

import akka.util.Timeout
import com.google.common.net.UrlEscapers
import play.api.libs.json.Json
import ru.yandex.tours.model.{HotelProvider, Source, TourOperator}
import ru.yandex.tours.util.http.AsyncHttpClient
import spray.http.Uri

import scala.concurrent.duration._
import scala.concurrent.{ExecutionContext, Future}

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 01.06.15
 *
 * Count for each search source amount of success and failures. If failures is too much(more than 50%) then search source
 * considered to be not alive. Statistic is gathered for 5 minutes.
 */
class SearchSourceAvailability[S <: Source](host: String, port: Int, httpClient: AsyncHttpClient, component: String)(implicit ec: ExecutionContext) {

  private implicit val timeout = Timeout(1.second)

  private val baseUrl = s"http://$host:$port/api/v2/clicker/tours/ru/$component"
  private val escaper = UrlEscapers.urlPathSegmentEscaper()

  private val body = Array.empty[Byte]

  private def objectId(tourOperator: Source, name: String) = {
    tourOperator.id + "-" + escaper.escape(name)
  }
  private def success(tourOperator: Source) = objectId(tourOperator, "success")
  private def failure(tourOperator: Source) = objectId(tourOperator, "failure")

  private def putUri(objectId: String): Uri = baseUrl + "/" + objectId
  private def getUri(objectIds: String*) = {
    Uri(baseUrl).withQuery(objectIds.map("id" -> _): _*)
  }

  def markSuccess(tourOperator: Source): Future[Unit] = {
    httpClient.put(putUri(success(tourOperator)), body).map(_ => ())
  }

  def markFailure(tourOperator: Source): Future[Unit] = {
    httpClient.put(putUri(failure(tourOperator)), body).map(_ => ())
  }

  def getAvailability(tourOperator: S): Future[Availability] = {
    val successId = success(tourOperator)
    val failureId = failure(tourOperator)
    httpClient.get(getUri(successId, failureId)).map {
      case ((status, response)) =>
        val json = Json.parse(response)
        Availability(
          (json \ successId).asOpt[Int].getOrElse(0),
          (json \ failureId).asOpt[Int].getOrElse(0)
        )
    }
  }

  def getAvailability(operators: Seq[S]): Future[Map[S, Availability]] = {
    val ids = operators.flatMap(o => Seq(success(o), failure(o)))
    httpClient.get(getUri(ids: _*)).map {
      case ((status, response)) =>
        val json = Json.parse(response)
        operators.map { operator =>
          operator -> Availability(
            (json \ success(operator)).asOpt[Int].getOrElse(0),
            (json \ failure(operator)).asOpt[Int].getOrElse(0)
          )
        }.toMap
    }
  }
}

case class Availability(successes: Int, failures: Int) {
  def total = successes + failures

  def isAlive = (total == 0) || (successes.toDouble / total > 0.5d)
}

class TourOperatorAvailability(host: String, port: Int, httpClient: AsyncHttpClient)(implicit ec: ExecutionContext) extends SearchSourceAvailability[TourOperator](host, port, httpClient, "tour_operator")

class HotelProvidersAvailability(host: String, port: Int, httpClient: AsyncHttpClient)(implicit ec: ExecutionContext) extends SearchSourceAvailability[HotelProvider](host, port, httpClient, "hotel_provider")