package ru.yandex.tours.parsing

import java.io.{FileInputStream, InputStream, InputStreamReader}
import java.util.zip.GZIPInputStream
import javax.xml.stream.{XMLInputFactory, XMLStreamConstants, XMLStreamReader}

import ru.yandex.tours.model.BaseModel.LangToVal
import ru.yandex.tours.model.Languages
import ru.yandex.tours.model.Languages.Lang
import ru.yandex.tours.model.util.proto
import ru.yandex.tours.util.Logging

import scala.collection.JavaConversions._
import scala.collection.mutable
import scala.util.{Failure, Success, Try}

abstract class AbstractCompanyXmlParser[T] extends Logging {
  private var errors = 0
  private var processed = 0
  private var parsedCompanies = mutable.Buffer.empty[T]

  def parseInputStream(is: InputStream): Iterable[T]

  def parseFile(filename: String): Iterable[T]

  protected def parseInputStream(is: InputStream, rootTag: String, objectTag: String) = {
    val factory = XMLInputFactory.newInstance
    val parser = factory.createXMLStreamReader(new InputStreamReader(is))
    while (parser.hasNext) {
      parser.next
      if (parser.isStartElement && parser.getLocalName.equalsIgnoreCase(rootTag)) {
        parseCompanies(parser, objectTag)
      }
    }
    log.info(s"Parse xml export finished. Objects parsed: ${parsedCompanies.size}. Errors $errors")
    parsedCompanies
  }

  protected def parseFile(filename: String, rootTag: String, objectTag: String) = {
    var fis: InputStream = new FileInputStream(filename)
    if (filename.endsWith(".gz")) {
      fis = new GZIPInputStream(fis)
    }
    parseInputStream(fis, rootTag, objectTag)
  }

  protected def parseTag(parser: XMLStreamReader, tag: String)(handler: PartialFunction[String, Unit]) = {
    assert(parser.getLocalName.equalsIgnoreCase(tag), s"${parser.getLocalName} found, but $tag is expected")
    while (parser.hasNext && !isNextEnd(parser, tag)) {
      if (parser.isStartElement) {
        val name = parser.getLocalName.toLowerCase
        handler.applyOrElse(name, (_: String) ⇒ ())
      }
    }
  }

  private def parseCompanies(parser: XMLStreamReader, objectTag: String) {
    while (parser.hasNext) {
      parser.next
      if (parser.isStartElement && parser.getLocalName.equalsIgnoreCase(objectTag)) {
        Try(parseCompany(parser)) match {
          case Success(Some(agency)) ⇒
            parsedCompanies += agency
          case Success(None) ⇒
          case Failure(e) ⇒
            errors += 1
            log.error(s"can not parse company: " + e.getMessage)
        }
      }
      processed += 1
      if (processed % 10000 == 0) {
        log.debug(s"$processed processed")
      }
    }
  }

  protected def parseCompany(parser: XMLStreamReader): Option[T]

  protected def parseString(parser: XMLStreamReader) = {
    val sb = new StringBuilder
    while (parser.next() == XMLStreamConstants.CHARACTERS) {
      sb.append(parser.getText.trim)
    }
    sb.toString()
  }

  protected def getOptAttribute(parser: XMLStreamReader, name: String) = {
    (0 until parser.getAttributeCount)
      .find(index ⇒ parser.getAttributeLocalName(index).equalsIgnoreCase(name))
      .map(parser.getAttributeValue)
  }

  protected def getAttribute(parser: XMLStreamReader, name: String) = {
    getOptAttribute(parser, name).getOrElse(sys.error(s"Can't find `$name` attribute in tag"))
  }

  protected def getOptLangAttribute(parser: XMLStreamReader): Option[Lang] = {
    getOptAttribute(parser, "lang").orElse(getOptAttribute(parser, "locale"))
      .flatMap(lang ⇒ Try(Languages.withName(lang)).toOption)
  }

  protected def getLangAttribute(parser: XMLStreamReader): Lang = {
    getOptLangAttribute(parser).getOrElse(Languages.other)
  }

  protected def isNextEnd(parser: XMLStreamReader, tag: String) = {
    parser.next == XMLStreamConstants.END_ELEMENT && parser.getLocalName.equalsIgnoreCase(tag)
  }

  protected def toLangVal(x: Iterable[(Lang, String)]): java.lang.Iterable[LangToVal] = {
    asJavaIterable(x.map { case (lang, value) ⇒ proto.toLangVal(lang.toString, value) })
  }
}
