package ru.yandex.tours.partners

import org.joda.time.LocalDate
import ru.yandex.tours.geo
import ru.yandex.tours.geo.base.region.Tree
import ru.yandex.tours.geo.mapping.GeoMappingHolder
import ru.yandex.tours.hotels.HotelsIndex
import ru.yandex.tours.model.BaseModel.Currency
import ru.yandex.tours.model.Languages.Lang
import ru.yandex.tours.model.hotels.{PartnerInfo, Partners}
import ru.yandex.tours.model.search.{ExtendedBaseRequest, ExtendedHotelSearchRequest, ExtendedOfferSearchRequest}
import ru.yandex.tours.model.util.Ages
import spray.http.Uri

import scala.util.{Failure, Try}

object BookingHttp {
  private val partner = Partners.booking
  val host = "https://distribution-xml.booking.com"
  val hotelResource = s"$host/json/bookings.getHotelAvailability"
  val blockResource = s"$host/json/bookings.getBlockAvailability"

  val affiliateId = "350687"

  def bookingLandingPage(hotelId: String, guestCount: Int, numberOfRooms: Int,
                  blockId: String, checkIn: LocalDate, nights: Int, lang: Lang): Uri = {
    Uri("https://secure.booking.com/book.html").withQuery(
      "aid" -> affiliateId,
      "hotel_id" -> hotelId,
      "stage" -> "1",
      "checkin" -> checkIn.toString,
      "interval" -> nights.toString,
      "lang" -> lang.toString,
      "label" -> "yandextravel",
      s"nr_rooms_$blockId" -> numberOfRooms.toString
    )
  }

  def hotelLandingPage(hotelUrl: String, guests: Ages, checkIn: LocalDate, nights: Int,
                  lang: Lang, currency: Currency, blockId: Option[String] = None, numberOfRooms: Option[Int] = None): Uri = {
    val query = Uri.Query.newBuilder
    var fragment = ""

    query ++= Seq(
      "aid" → affiliateId,
      "checkin" → checkIn.toString,
      "interval" → nights.toString,
      "lang" → lang.toString,
      "selected_currency" → currency.name,
      "group_adults" → guests.adults.toString,
      "label" → "yandextravel"
    )

    if (guests.kids > 0) {
      query += "group_children" → guests.kids.toString
      for (age ← guests.kidsAges)  {
        query += "age" → age.toString
      }
    }
    if (blockId.isDefined) {
      query += "show_room" -> blockId.get
      fragment = "RD" + blockId.get.split("_").head
    }
    if (numberOfRooms.isDefined) {
      query += "no_rooms" -> numberOfRooms.get.toString
    }

    val uri = Uri(hotelUrl).withQuery(query.result())
    if (fragment.isEmpty) {
      uri
    } else {
      uri.withFragment(fragment)
    }
  }

  private def hotelAvailability(guests: Ages,
                                chunk: Int,
                                checkIn: LocalDate, checkOut: LocalDate,
                                lang: Lang, currency: Currency,
                                additionalParams: (String, String)*): Uri = {
    val withoutInfants = guests.guestsWithoutInfants
    var params = Map(
      "arrival_date" -> checkIn.toString,
      "departure_date" -> checkOut.toString,
      "currency_code" -> currency.name(),
      "max_persons" -> (-withoutInfants).toString,
      "guest_qty" -> withoutInfants.toString,
      "children_qty" → guests.kids.toString,
      "children_age" → guests.kidsAges.mkString(","),
      "show_price_room_info" -> "1",
      "available_rooms" -> "1",
      "languagecode" -> lang.toString
    )
    if (chunk > 0) {
      params = params ++: Map("chunk" -> chunk.toString)
    }
    params = params ++: additionalParams.toMap
    Uri(hotelResource).withQuery(params)
  }

  def hotelAvailabilityByCity(cityId: String,
                              guests: Ages,
                              chunk: Int,
                              checkIn: LocalDate, checkOut: LocalDate,
                              lang: Lang, currency: Currency): Uri = {
    hotelAvailability(guests, chunk, checkIn, checkOut, lang, currency, "city_ids" -> cityId)
  }

  def hotelAvailabilityByRegion(regionId: String, guests: Ages, chunk: Int,
                                checkIn: LocalDate, checkOut: LocalDate,
                                lang: Lang, currency: Currency): Uri = {
    hotelAvailability(guests, chunk, checkIn, checkOut, lang, currency, "region_ids" -> regionId)
  }
  def hotelAvailabilityByHotelIds(hotelIds: List[String], guests: Ages, chunk: Int,
                                  checkIn: LocalDate, checkOut: LocalDate,
                                  lang: Lang, currency: Currency): Uri = {
    hotelAvailability(guests, chunk, checkIn, checkOut, lang, currency, "hotel_ids" -> hotelIds.mkString(","))
  }

  def hotelAvailabilityByArea(latitude: Double, longitude: Double, radiusInM: Int, guests: Ages, chunk: Int,
                              checkIn: LocalDate, checkOut: LocalDate,
                              lang: Lang, currency: Currency): Uri = {
    hotelAvailability(guests, chunk, checkIn, checkOut, lang, currency,
      "longitude" -> longitude.toString,
      "latitude" -> latitude.toString,
      "radius" -> radiusInM.toString)
  }

  private val goodQualityPrefix = "http://q-ec.bstatic.com/images/hotel/840x460"
  private val badQualityPrefix = "http://bstatic.com/images/hotel/org"

  def increasePhotoQuality(url: String): String = {
    url.replace(badQualityPrefix, goodQualityPrefix)
  }

  def decreasePhotoQuality(url: String): String = {
    url.replace(goodQualityPrefix, badQualityPrefix)
  }

  def blockAvailability(hotelId: String, checkIn: LocalDate, checkOut: LocalDate,
                        lang: Lang, currency: Currency): Uri = {
    Uri(blockResource).withQuery(
      "arrival_date" -> checkIn.toString,
      "departure_date" -> checkOut.toString,
      "detail_level" -> "1",
      "hotel_ids" -> hotelId,
      "currency_code" -> currency.name(),
      "show_meta_mealplans" -> "1",
      "languagecode" -> lang.toString
    )
  }

  def hotel2region(offset: Int, pageSize: Int): String = {
    s"https://distribution-xml.booking.com/json/bookings.getHotels?rows=$pageSize&offset=$offset&fields=hotel_id,city_id"
  }

  def getUri(searchRequest: ExtendedBaseRequest,
             hotelsIndex: HotelsIndex,
             geoMappingHolder: GeoMappingHolder,
             tree: Tree): Try[Uri] = searchRequest match {
    case r: ExtendedHotelSearchRequest => getHotelsUri(r, hotelsIndex, geoMappingHolder, tree, 0)
    case r: ExtendedOfferSearchRequest => getBlocksUri(r, hotelsIndex)
    case _ => Failure(new Exception(s"Can not build booking url for request $searchRequest"))
  }

  def getHotelsUri(searchRequest: ExtendedHotelSearchRequest,
                   hotelsIndex: HotelsIndex,
                   geoMapping: GeoMappingHolder,
                   tree: Tree,
                   chunk: Int): Try[Uri] = Try {
    val geoId = searchRequest.to
    val checkIn = searchRequest.when
    val checkOut = searchRequest.when.plusDays(searchRequest.nights)
    val guests = searchRequest.hotelSearchRequest.getAges

    searchRequest.hotelIds match {
      case Some(ids) =>
        val hotelIds = hotelsIndex.getHotelsById(ids).values.flatMap(_.partnerId(partner)).toList
        BookingHttp.hotelAvailabilityByHotelIds(hotelIds, guests, chunk, checkIn, checkOut,
          searchRequest.lang, searchRequest.currency)
      case _ =>
        geoMapping.getPartnerCity(partner, geoId).orElse(geoMapping.getPartnerCountry(partner, geoId)) match {
          case Some(region) =>
            if (region.startsWith("r")) {
              BookingHttp.hotelAvailabilityByRegion(region.substring(1), guests, chunk, checkIn, checkOut,
                searchRequest.lang, searchRequest.currency)
            } else {
              BookingHttp.hotelAvailabilityByCity(region, guests, chunk, checkIn, checkOut,
                searchRequest.lang, searchRequest.currency)
            }
          case None =>
            tree.region(searchRequest.to) match {
              case Some(region) =>
                val lonCenter = region.longitude
                val latCenter = region.latitude
                var radius = (geo.distanceInKm(
                  latCenter,
                  lonCenter,
                  latCenter + region.boundingBox.latSpan,
                  lonCenter + region.boundingBox.lonSpan
                ) * 1000).toInt
                if (radius == 0) radius = 10
                BookingHttp.hotelAvailabilityByArea(latCenter, lonCenter, radius, guests, chunk, checkIn, checkOut,
                  searchRequest.lang, searchRequest.currency)
              case None => throw new Exception(s"Unknown geoId $geoId by Booking")
            }
        }
    }
  }

  def getBlocksUri(offerSearchRequest: ExtendedOfferSearchRequest, hotelsIndex: HotelsIndex): Try[Uri] = Try {
    val ourHotelId = offerSearchRequest.hotelId
    val hotelId = (for {
      hotel <- hotelsIndex.getHotelById(ourHotelId)
      PartnerInfo(_, bookingId, _, _, _) <- hotel.partnerIds.find(_.partner == partner)
    } yield bookingId).getOrElse(sys.error(s"Unknown hotel $ourHotelId by Booking"))
    val checkIn = offerSearchRequest.hotelRequest.when
    val checkOut = checkIn.plusDays(offerSearchRequest.hotelRequest.nights)
    BookingHttp.blockAvailability(hotelId, checkIn, checkOut,
      offerSearchRequest.hotelRequest.lang, offerSearchRequest.hotelRequest.currency)
  }
}
