package ru.yandex.tours.partners

import org.apache.commons.codec.digest.DigestUtils
import ru.yandex.tours.geo.mapping.GeoMappingHolder
import ru.yandex.tours.hotels.HotelsIndex
import ru.yandex.tours.model.hotels.Partners
import ru.yandex.tours.model.search.{ExtendedBaseRequest, ExtendedHotelSearchRequest, ExtendedOfferSearchRequest}
import ru.yandex.tours.util.ip.RandomIpGenerator
import spray.http.Uri

import scala.concurrent.duration._
import scala.util.hashing.MurmurHash3
import scala.util.{Failure, Success, Try}

/**
  * Created by asoboll on 29.11.16.
  */
object HotelsCombinedHttp {
  val partner = Partners.hotelsCombined

  // real api:
  private val apiRoot = "https://166338.api.hotelscombined.com/api/2.0"
  // test: private val apiRoot = "http://test.166338.api.hotelscombined.com/api/2.0"

  val authHeaders = List(
    ("Accept", "application/x-protobuf"),
    ("User-Agent", "ning/1.x (x86_64-pc-linux-gnu)")
  )

  // OJSC Rostelecom block 178.64-71.*.*, 524288 total
  private val rostelecomSubnet = RandomIpGenerator.Subnet(0xb2400000, 0xfff80000)
  //                       101.112-9.*.*, 524288 total
  private val someAurstralianSubnet = RandomIpGenerator.Subnet(0x65700000, 0xfff80000)
  private val ipGenerator = RandomIpGenerator.create(someAurstralianSubnet)

  private val sessionTimeframe = 20.minutes.toMillis

  private val PAGE_SIZE = 100 // 100 max


  private def sessionParams(request: ExtendedBaseRequest, apiKey: String) = {
    val time = System.currentTimeMillis() / sessionTimeframe
    val sessionId = DigestUtils.md5Hex(request.hotelRequest.encoded + time.toString)
    val ip = ipGenerator.fromInt(MurmurHash3.stringHash(sessionId))
    Seq(
      "apiKey" -> apiKey,
      "sessionId" -> sessionId,
      "onlyIfComplete" -> "false",
      "includeTaxesInTotal" -> "true",
      "includeLocalTaxesInTotal" -> "false",
      "languageCode" -> "RU",
      "currencyCode" -> "RUB",
      "clientIp" -> ip.getHostAddress
      //"searchTimeout" -> ???
    )
  }

  private def queryDetailParams(request: ExtendedBaseRequest, destination: Option[String]) = {
    val checkIn = request.hotelRequest.when
    val checkOut = checkIn.plusDays(request.hotelRequest.nights)
    val adults = request.hotelRequest.adults
    val children = request.hotelRequest.kidsAges
    Seq(
      "checkin" -> checkIn.toString,
      "checkout" -> checkOut.toString,
      "rooms" -> (adults + (if (children.nonEmpty) children.mkString(":", ",", "") else ""))
    ) ++ destination.map(d => "destination" -> s"place:$d")
  }

  private val multipleHotelParams = {
    Seq(
      "pageSize" -> s"$PAGE_SIZE",
      "responseOptions" -> "toprates",
      "sortField" -> "minRate",
      "sortDirection" -> "ascending"
    )
  }

  def multipleHotelRequest(request: ExtendedHotelSearchRequest,
                           hotelsIndex: HotelsIndex,
                           geoMapping: GeoMappingHolder,
                           apiKey: String
                          ): Try[Uri] = Try {
    val destination = geoMapping.getPartnerDestination(partner, request.to)
      .getOrElse(sys.error(s"Unknown destination ${request.to} by $partner"))

    val query = Uri.Query.newBuilder
    query ++= sessionParams(request, apiKey)
    query ++= queryDetailParams(request, Some(destination))
    query ++= multipleHotelParams
    request.hotelIds.foreach { hotelIds =>
      val ids = hotelIds.flatMap(getPartnerHotelIds(_, hotelsIndex))
      if (ids.nonEmpty) query += "preferredHotels" -> ids.take(50).mkString("|")
    }
    Uri(apiRoot + "/hotels").withQuery(query.result())
  }

  implicit class RichHCUri(uri: Uri) {
    def pageIndex: Int = uri.query.get("pageIndex").fold(0)(_.toInt)

    def withPage(page: Int): Uri = uri.withQuery(uri.query.toMap.updated("pageIndex", s"$page"))

    def nextPage: Uri = withPage(pageIndex + 1)

    def lastRateIndex: Int = (pageIndex + 1) * PAGE_SIZE

    def withNoDestination: Uri = uri.withQuery(uri.query.toMap - "destination")
  }

  def singleHotelRequest(request: ExtendedOfferSearchRequest,
                         hotelsIndex: HotelsIndex,
                         geoMapping: GeoMappingHolder,
                         id2key: Map[String, String],
                         apiKey: String): Iterable[Try[Uri]] = {
    Try {
      val destinationOpt = geoMapping.getPartnerCity(partner, request.hotelRequest.to)
      val params = sessionParams(request, apiKey) ++ queryDetailParams(request, destinationOpt)
      def buildUriForPartnerId(id: String) = Try {
        val partnerKey = id2key.getOrElse(id,
          throw new Exception(s"Unknown hotelKey for id $id by $partner, hotel ${request.hotelId}"))
        val query = Uri.Query.newBuilder ++= params += "hotel" -> s"hotel:$partnerKey"
        Uri(apiRoot + "/hotel").withQuery(query.result())
      }
      getPartnerHotelIds(request.hotelId, hotelsIndex).map(buildUriForPartnerId)
    } match {
      case Success(it) if it.nonEmpty => it
      case Success(_) => Iterable(Failure(new Exception(s"Unknown hotel ${request.hotelId} by $partner")))
      case Failure(e) => Iterable(Failure(e))
    }
  }

  private def getPartnerHotelIds(hotelId: Int, hotelsIndex: HotelsIndex): Iterable[String] = {
    for {
      hotel <- hotelsIndex.getHotelById(hotelId).toIterable
      partnerId <- hotel.partnerId(partner)
    } yield partnerId
  }

}
