package ru.yandex.tours.partners

import java.io._
import java.nio.{ByteBuffer, MappedByteBuffer}

import com.google.protobuf.ByteString
import org.apache.commons.io.IOUtils
import ru.yandex.extdata.common.exception.{DataIsNotLoadedException, UnableToLoadDataException}
import ru.yandex.extdata.common.meta.DataType
import ru.yandex.extdata.common.service.ExtDataService
import ru.yandex.tours.extdata.{DataDef, DataTypes}
import ru.yandex.tours.util.collections.ByteBufferMappedMap.{MappedMap, MappedMapWriter}
import ru.yandex.tours.util.collections.MapWrapper
import ru.yandex.tours.util.io.ByteBuffers
import ru.yandex.tours.util.{IO, Logging}

import scala.concurrent.duration.{FiniteDuration, _}
import scala.util.Try
import scala.util.control.NonFatal

/**
  * Created by asoboll on 16.12.16.
  */
object HotelsCombinedId2Key extends DataDef[MappedHotelsCombinedId2Key] with Logging {

  val dataType: DataType = DataTypes.hcId2Key

  override def from(extDataService: ExtDataService): MappedHotelsCombinedId2Key = {
    try {
      super.from(extDataService)
    } catch {
      case _: DataIsNotLoadedException | _: UnableToLoadDataException =>
        log.warn(s"No previous ${dataType.getName}. Using empty one.")
        MappedHotelsCombinedId2Key.empty
    }
  }

  override def parse(is: InputStream): MappedHotelsCombinedId2Key = {
    val file = IO.usingTmp(dataType.getName) { os => IOUtils.copy(is, os) }
    val id2key = MappedHotelsCombinedId2Key.fromFile(file)
    id2key.load()
    id2key
  }

  override protected def onChange(oldValue: Option[MappedHotelsCombinedId2Key],
                                  newValue: MappedHotelsCombinedId2Key): Unit = {
    oldValue.foreach(_.closeAfter(1.minute))
    oldValue.foreach(_.delete())
  }
}

class MappedHotelsCombinedId2Key(id2key: Map[String, String],
                                 buffer: Option[MappedByteBuffer],
                                 file: Option[File])
  extends MapWrapper(id2key) with Closeable {

  def load() = buffer.foreach(_.load())

  def delete() = file.foreach(_.delete())

  def close(): Unit = buffer.foreach(ByteBuffers.unmap)

  def closeAfter(interval: FiniteDuration): Unit = {
    IO.closeAfter(this, interval)
  }
}

object MappedHotelsCombinedId2Key extends Logging {
  private val formatVersion = 1
  private val headerSize = 2 * Integer.BYTES

  implicit val serializeString = (s: String) => s.getBytes
  implicit val deserializeString = (bb: ByteBuffer) => new String(ByteString.copyFrom(bb).toByteArray)

  def empty: MappedHotelsCombinedId2Key = new MappedHotelsCombinedId2Key(Map.empty, None, None)

  private def parseBuffer(buffer: ByteBuffer): MappedMap[String, String] = {
    val version = buffer.getInt
    require(version == formatVersion, s"Unknown format version: $version")
    val size = buffer.getInt
    val map = MappedMap.from[String, String](buffer.slice)
    require(size == -1 || size == map.size, s"Record count ${map.size} doesnt match header: $size")
    map
  }

  def fromFile(file: File): MappedHotelsCombinedId2Key = {
    Try {
      val buffer = ByteBuffers.mmap(file)
      val id2key = parseBuffer(buffer)
      new MappedHotelsCombinedId2Key(id2key, Some(buffer), Some(file))
    }.recover {
      case NonFatal(t) =>
        log.error("Cannot parse hotels combined id2key", t)
        MappedHotelsCombinedId2Key.empty
    }.get
  }

  def writeTo(os: OutputStream, id2key: TraversableOnce[(String, String)]): Unit = {
    val mapWriter = new MappedMapWriter[String, String](os)

    val header = ByteBuffer.allocate(headerSize)
      .putInt(formatVersion)
      .putInt(id2key.size)
    os.write(header.array())

    mapWriter.write(id2key.toSeq.sortBy(_._1))

    mapWriter.finish()
    os.close()
  }

  def writeTo(file: File, id2Key: TraversableOnce[(String, String)]): Unit = {
    val os = new BufferedOutputStream(new FileOutputStream(file))
    writeTo(os, id2Key)
  }
}
