package ru.yandex.tours.partners

import java.net.URLEncoder

import org.joda.time.LocalDate
import org.joda.time.format.DateTimeFormat
import ru.yandex.tours.geo.base.region.Tree
import ru.yandex.tours.geo.mapping.GeoMappingHolder
import ru.yandex.tours.model.hotels.Partners
import ru.yandex.tours.model.search.{ExtendedHotelSearchRequest, HotelSearchFilter, SearchFilter}

import scala.util.Try

object LtHttp {
  private val dateFormat = DateTimeFormat.forPattern("dd.MM.yyyy")

  val host = "https://api.level.travel"

  val destinations = host + "/references/destinations"

  val hotelsDump = s"$host/references/hotels?full_dump=true&api_version=3"

  def hotel(hotelId: Int): String = s"$host/references/hotels?hotel_ids=$hotelId"

  def hotelsByRegion(regionId: Int): String = {
    s"$host/references/hotels?region_ids=$regionId"
  }

  def actualization(requestId: String, tourId: String): String = {
    s"$host/search/actualize?request_id=${encoded(requestId)}&tour_id=${encoded(tourId)}"
  }

  def tours(hotelId: String, requestId: String, operatorIds: Iterable[String]): String = {
    s"$host/search/get_hotel_offers?compact=true&hotel_id=$hotelId&request_id=${encoded(requestId)}" +
      s"&operator_ids=${encoded(operatorIds.mkString(","))}"
  }

  def result(requestId: String, operatorIds: Iterable[String]): String = {
    s"$host/search/results?compact=true&request_id=${encoded(requestId)}" +
      s"&operator_ids=${encoded(operatorIds.mkString(","))}"
  }

  def status(requestId: String): String = {
    s"$host/search/status?request_id=${encoded(requestId)}"
  }

  def tourLink(tourId: String, requestId: String): String = {
    s"https://level.travel/package_details/$tourId?request_id=${encoded(requestId)}&aflt=Ya"
  }

  def enqueue(fromCity: String, fromCountry: String,
              toCountry: String,
              nightsFrom: Int, nightsTo: Int,
              adults: Int,
              when: LocalDate, kids: Iterable[Int], flexDates: Boolean,
              hotelId: Option[String], toCity: Option[String],
              stars: Range): String = {
    var params = Map(
      "from_city" -> fromCity,
      "to_country" -> toCountry,
      "nights" -> s"$nightsFrom..$nightsTo",
      "adults" -> adults.toString,
      "start_date" -> dateFormat.print(when),
      "kids" -> kids.size.toString,
      "stars_from" -> stars.head.toString,
      "stars_to" -> stars.last.toString,
      "flex_dates" -> flexDates.toString,
      "from_country" -> fromCountry
    )
    hotelId.foreach(id => params += "hotel_ids" -> id)
    if (kids.nonEmpty) {
      params += "kids_ages" -> kids.mkString(",")
    }
    toCity.foreach(city => params += "to_city" -> city)
    val cgiParams = params.map(t => s"${t._1}=${encoded(t._2)}").mkString("&")
    s"$host/search/enqueue?$cgiParams"
  }

  private def encoded(param: String) = {
    URLEncoder.encode(param, "UTF-8")
  }

  private val partner = Partners.lt

  def starsRange(filter: SearchFilter): Range = {
    filter match {
      case f: HotelSearchFilter => f.starsFrom to f.starsTo
      case _ => 1 to 5
    }
  }

  def createEnqueueUrl(request: ExtendedHotelSearchRequest, tree: Tree,
                       geoMapping: GeoMappingHolder,
                       hotelId: Option[String]): Try[String] = Try {
    val region = tree.region(request.to).getOrElse(sys.error(s"Region not found: ${request.to}"))
    val knownRegion = tree.pathToRoot(region)
      .find(r => geoMapping.isKnownDestination(r.id))
      .getOrElse(sys.error(s"No known destination parent for ${region.id}"))

    val country = tree.country(knownRegion).getOrElse(sys.error(s"No parent of type Country for region ${request.to}"))

    val city = if (hotelId.isEmpty) geoMapping.getPartnerCity(partner, knownRegion.id) else Option.empty
    if (country != knownRegion && hotelId.isEmpty) require(city.isDefined, s"No city mapping for region ${region.id}")

    val fromCountry = tree.country(request.from).getOrElse(sys.error(s"No parent of type Country for region ${request.from}"))
    val from = geoMapping.getPartnerDeparture(partner, request.from).getOrElse(sys.error(s"No mapping for region: ${request.from}"))
    LtHttp.enqueue(
      from,
      geoMapping.getPartnerCountry(partner, fromCountry.id).getOrElse(sys.error(s"No mapping for region ${fromCountry.id}")),
      geoMapping.getPartnerCountry(partner, country.id).getOrElse(sys.error(s"No mapping for region ${country.id}")),
      request.baseRequest.nightsRange.min,
      request.baseRequest.nightsRange.max,
      request.baseRequest.adults,
      request.when,
      request.baseRequest.kidsAges,
      request.flexWhen,
      hotelId,
      city,
      starsRange(request.filter)
    )
  }
}
