package ru.yandex.tours.partners

import org.joda.time.LocalDate
import play.api.libs.json._
import ru.yandex.tours.geo.mapping.GeoMappingHolder
import ru.yandex.tours.hotels.HotelsIndex
import ru.yandex.tours.model.hotels.Partners
import ru.yandex.tours.model.search.{ExtendedBaseRequest, ExtendedHotelSearchRequest, ExtendedOfferSearchRequest}
import ru.yandex.tours.util.parsing.IntValue
import spray.http.Uri

import scala.util.{Failure, Success, Try}

object OstrovokV3Http {
  val host = "https://dapi.worldota.net"
  private val rateResource = s"$host/api/affiliate/v3/hotel/rates"

  private def commonParams(adults: Int,
                           checkIn: LocalDate, checkOut: LocalDate,
                           children: Iterable[Int],
                           page: Option[Int]) = {
    Json.obj(
      "checkin" -> checkIn.toString,
      "checkout" -> checkOut.toString,
      "adults" -> adults,
      "children" -> children,
      "currency" -> "RUB",
      "limit" -> 300
    ) ++ JsObject(page.map(p => "page" -> JsNumber(p)).toSeq)
  }

  private def uri(params: JsObject) = {
    Uri(rateResource).withQuery(("data", params.toString()))
  }

  def hotelRates(regionId: Int, adults: Int,
                 checkIn: LocalDate, checkOut: LocalDate,
                 children: Iterable[Int],
                 page: Option[Int]): Uri = {
    uri(commonParams(adults, checkIn, checkOut, children, page) + ("region_id" -> JsNumber(regionId)))
  }

  def hotelRates(hotelIds: Iterable[String], adults: Int,
                 checkIn: LocalDate, checkOut: LocalDate,
                 children: Iterable[Int],
                 page: Option[Int]): Uri = {
    val params = commonParams(adults, checkIn, checkOut, children, page) +
      ("ids" -> JsArray(hotelIds.map(JsString).toSeq))
    uri(params)
  }

  private val partner = Partners.ostrovokv3

  def getSearchUri(request: ExtendedBaseRequest, hotelsIndex: HotelsIndex, geoMapping: GeoMappingHolder): Try[Uri] = {
    val checkIn = request.hotelRequest.when
    val checkOut = checkIn.plusDays(request.hotelRequest.nights)
    val adults = request.hotelRequest.adults
    val children = request.hotelRequest.kidsAges
    request match {
      case ExtendedHotelSearchRequest(_, _, Some(hotelIds)) =>
        Try {
          val ostrovokIds = hotelsIndex.getHotelsById(hotelIds).values.flatMap(_.partnerId(partner))
          OstrovokV3Http.hotelRates(ostrovokIds, adults, checkIn, checkOut, children, None)
        }
      case hotelRequest: ExtendedHotelSearchRequest =>
        for {
          region <- getRegion(hotelRequest.to, geoMapping)
        } yield OstrovokV3Http.hotelRates(region, adults, checkIn, checkOut, children, None)
      case tourRequest: ExtendedOfferSearchRequest =>
        val ids = for {
          hotel <- hotelsIndex.getHotelById(tourRequest.hotelId).toIterable
          ostrovokId <- hotel.partnerId(partner)
        } yield ostrovokId
        if (ids.isEmpty) {
          Failure(new Exception(s"OstrovokV3 doesn't know about hotel with id: ${tourRequest.hotelId}"))
        } else {
          Success(OstrovokV3Http.hotelRates(ids, adults, checkIn, checkOut, children, None))
        }
      case req => Failure(new Exception(s"OstrovokV3 searcher doesn't support such request: $req"))
    }
  }

  private def getRegion(geoId: Int, geoMapping: GeoMappingHolder): Try[Int] = {
    geoMapping.getPartnerCity(partner, geoId).orElse(geoMapping.getPartnerCountry(partner, geoId)) match {
      case Some(IntValue(ostrovokRegion)) => Success(ostrovokRegion)
      case Some(notIntValue) => Failure(new Exception(s"OstrovokV3 regions expected to be int: $notIntValue"))
      case None => Failure(new Exception(s"Unknown geoId $geoId by OstrovokV3"))
    }
  }
}
