package ru.yandex.tours.partners

import com.typesafe.config.Config
import ru.yandex.tours.model.hotels.Partners
import ru.yandex.tours.model.hotels.Partners.Partner
import ru.yandex.tours.model.search.SearchType
import ru.yandex.tours.model.search.SearchType.SearchType
import ru.yandex.tours.util.Decrypter

import scala.collection.JavaConverters._
import scala.util.Try

object PartnerConfigs {
  def parse(config: Config): Map[Partner, PartnerConfig] = {
    val Array(key, iv) = config.getString("tours.partners.secret").split(':')
    val decrypter = new Decrypter(key, iv)
    val result = for {
      partner <- Partners.values.toIterable
      conf <- Try(config.getConfig(s"partners.$partner")).toOption.toIterable
    } yield partner -> PartnerConfig.fromConfig(partner, conf, decrypter)
    result.toMap
  }
}

object PartnerConfig {
  def fromConfig(partner: Partner, conf: Config, crypto: Decrypter): PartnerConfig = {
    val feedUrl = conf.getString("hotels.feed.url")
    val isExternalUrl = conf.getBoolean("hotels.feed.external")
    val gunzip = parseBooleanWithDefault(conf, "hotels.feed.gunzip")
    val headers = for {
      headerConf <- Try(conf.getConfig("headers")).toOption.toIterable
      entry <- headerConf.entrySet().asScala
      key = entry.getKey
    } yield key -> crypto.tryDecryptSection(key, headerConf.getString(key))

    val useCommonClient = parseBooleanWithDefault(conf, "use_common_client")
    val searchUrl = Try(conf.getString("search.url")).toOption
    val actualizeUrl = Try(conf.getString("search.actualize_url")).toOption
    val searchParameters = if (conf.hasPath("search")) Some(parseSearchParameters(conf.getConfig("search"), crypto)) else None
    PartnerConfig(
      partner,
      feedUrl,
      isExternalUrl,
      gunzip,
      useCommonClient,
      searchUrl,
      actualizeUrl,
      headers.toList,
      searchParameters
    )
  }

  private def parseSearchParameters(conf: Config, crypto: Decrypter) = {
    val searchType = SearchType.withName(conf.getString("type").toUpperCase)
    val headers = for {
      headerConf <- Try(conf.getConfig("headers")).toOption.toIterable
      entry <- headerConf.entrySet().asScala
      key = entry.getKey
    } yield key -> crypto.tryDecryptSection(key, headerConf.getString(key))

    val apiKey = if (conf.hasPath("ApiKey")) Option(crypto.decrypt(conf.getString("ApiKey"))) else Option.empty

    SearchParameters(
      searchType,
      parseBooleanWithDefault(conf, "by_span"),
      parseBooleanWithDefault(conf, "is_billing_important"),
      parseBooleanWithDefault(conf, "country_search"),
      parseBooleanWithDefault(conf, "is_departure_important"),
      parseBooleanWithDefault(conf, "use_smart_fallback", default = true),
      parseBooleanWithDefault(conf, "use_search_by_hotel_ids", default = true),
      parseBooleanWithDefault(conf, "explicit_region_type"),
      headers.toList,
      apiKey
    )
  }

  private def parseBooleanWithDefault(config: Config, path: String, default: Boolean = false) = {
    if (config.hasPath(path)) config.getBoolean(path) else default
  }
}

case class PartnerConfig(partner: Partner,
                         feedUrl: String,
                         isExternalUrl: Boolean,
                         gunzipFeed: Boolean,
                         useCommonClient: Boolean,
                         searchUrl: Option[String],
                         actualizeUrl: Option[String],
                         headers: List[(String, String)],
                         searchParameters: Option[SearchParameters]) {


  def getSearchHeaders: List[(String, String)] = {
    searchParameters.map(sp => if (sp.headers.nonEmpty) sp.headers else headers).orElse(Option(headers)).get
  }
}

case class SearchParameters(searchType: SearchType,
                            searchBySpan: Boolean = false,
                            isBillingImportant: Boolean = false,
                            countrySearch: Boolean = false,
                            isDepartureImportant: Boolean = false,
                            useSmartFallback: Boolean = true,
                            useSearchByHotelIds: Boolean = true,
                            explicitRegionType: Boolean = false,
                            headers: List[(String, String)],
                            apiKey: Option[String])
