package ru.yandex.tours.prices

import java.util.concurrent.atomic.AtomicReference

import org.joda.time.format.DateTimeFormat
import org.joda.time.{DateTime, LocalDate}
import ru.yandex.tours.clickhouse.ClickHouseClient
import ru.yandex.tours.geo.base.region.Tree
import ru.yandex.tours.geo.mapping.GeoMappingHolder
import ru.yandex.tours.hotels.HotelsIndex
import ru.yandex.tours.model.search.GetOfferRequest
import ru.yandex.tours.model.search.SearchResults.ActualizedOffer
import ru.yandex.tours.util.Logging
import ru.yandex.tours.util.lang.Dates._
import ru.yandex.tours.util.parsing.Tabbed

import scala.collection.JavaConversions._
import scala.collection.mutable.ArrayBuffer
import scala.concurrent.duration._
import scala.concurrent.{Await, ExecutionContext}
import scala.util.control.NonFatal

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 17.08.15
 */
class ActualizationStorage(clickHouseClient: ClickHouseClient,
                           hotelsIndex: HotelsIndex,
                           tree: Tree,
                           geoMapping: GeoMappingHolder)
                          (implicit ec: ExecutionContext) extends Logging {

  private val bufferSize = 200000
  private val flushAfter = bufferSize - 1000

  private val buffer = new AtomicReference(new ArrayBuffer[Record](bufferSize))

  clickHouseClient.update(
    s"""create table if not exists ${clickHouseClient.database}.actualized (
      |  date Date,
      |  timestamp DateTime,
      |  tour_id String,
      |
      |  from UInt32,
      |  hotel_id UInt32,
      |  hotel_geo_id UInt32,
      |  hotel_stars UInt8,
      |  when Date,
      |  nights UInt8,
      |  ages String,
      |  ages_array Array(UInt8),
      |
      |  pansion Int8,
      |  original_room_code String,
      |
      |  price Int32,
      |  actualized_price Int32,
      |  fuel_charge Int32,
      |  visa_price Int32,
      |  infant_price Int32,
      |
      |  flights Nested
      |  (
      |    additional_price Int32,
      |
      |    company String,
      |    flight_number String,
      |    from_geo_id UInt32,
      |    from_airport String,
      |    from_airport_code String,
      |    to_geo_id UInt32,
      |    to_airport String,
      |    to_airport_code String,
      |
      |    back_company String,
      |    back_flight_number String,
      |    back_from_geo_id UInt32,
      |    back_from_airport String,
      |    back_from_airport_code String,
      |    back_to_geo_id UInt32,
      |    back_to_airport String,
      |    back_to_airport_code String
      |  )
      |
      |) ENGINE = MergeTree(date, (timestamp, tour_id), 8192)
    """.stripMargin
  ).onFailure {
    case NonFatal(t) => log.warn("Failed to create table", t)
  }

  private val dateFormat = DateTimeFormat.forPattern("yyyy-MM-dd")
  private val dateTimeFormat = DateTimeFormat.forPattern("yyyy-MM-dd HH:mm:ss")
  private def escape(str: String) = str
    .replaceAllLiterally("\\", "\\\\")
    .replaceAllLiterally("\n", "\\n")
    .replaceAllLiterally("\t", "\\t")
    .replaceAllLiterally("'", "\\'")
    .replaceAllLiterally("\"", "\\\"")

  private def quote(str: String) = "'" + str + "'"

  private case class Flight(additionalPrice: Int,
                            company: String,
                            flightNumber: String,
                            fromGeoId: Int,
                            fromAirport: String,
                            fromAirportCode: String,
                            toGeoId: Int,
                            toAirport: String,
                            toAirportCode: String,
                            backCompany: String,
                            backFlightNumber: String,
                            backFromGeoId: Int,
                            backFromAirport: String,
                            backFromAirportCode: String,
                            backToGeoId: Int,
                            backToAirport: String,
                            backToAirportCode: String)

  private case class Record(timestamp: DateTime,
                            tourId: String,
                            from: Int,
                            hotelId: Int,
                            when: LocalDate,
                            nights: Int,
                            ages: Array[Int],
                            pansionId: Int,
                            roomCode: String,
                            price: Int,
                            actualizedPrice: Int,
                            fuelCharge: Int,
                            visaPrice: Option[Int],
                            infantPrice: Option[Int],
                            flights: Array[Flight]) {
    def toRowString: String = {
      val hotel = hotelsIndex.getHotelById(hotelId)
      val region = hotel.map(_.geoId)

      Tabbed(
        timestamp.toString(dateFormat),
        timestamp.toString(dateTimeFormat),
        escape(tourId),
        from,
        hotelId,
        region.getOrElse(0),
        hotel.fold(0)(_.star.id),
        when.toString(dateFormat),
        nights,
        ages.sorted.mkString(","),
        ages.sorted.mkString("[",",","]"),
        pansionId,
        escape(roomCode),
        price,
        actualizedPrice,
        fuelCharge,
        visaPrice.getOrElse(-1),
        infantPrice.getOrElse(-1),
        flights.map(_.additionalPrice).mkString("[",",","]"),
        flights.map(_.company).map(escape).map(quote).mkString("[",",","]"),
        flights.map(_.flightNumber).map(escape).map(quote).mkString("[",",","]"),
        flights.map(_.fromGeoId).mkString("[",",","]"),
        flights.map(_.fromAirport).map(escape).map(quote).mkString("[",",","]"),
        flights.map(_.fromAirportCode).map(escape).map(quote).mkString("[",",","]"),
        flights.map(_.toGeoId).mkString("[",",","]"),
        flights.map(_.toAirport).map(escape).map(quote).mkString("[",",","]"),
        flights.map(_.toAirportCode).map(escape).map(quote).mkString("[",",","]"),
        flights.map(_.backCompany).map(escape).map(quote).mkString("[",",","]"),
        flights.map(_.backFlightNumber).map(escape).map(quote).mkString("[",",","]"),
        flights.map(_.backFromGeoId).mkString("[",",","]"),
        flights.map(_.backFromAirport).map(escape).map(quote).mkString("[",",","]"),
        flights.map(_.backFromAirportCode).map(escape).map(quote).mkString("[",",","]"),
        flights.map(_.backToGeoId).mkString("[",",","]"),
        flights.map(_.backToAirport).map(escape).map(quote).mkString("[",",","]"),
        flights.map(_.backToAirportCode).map(escape).map(quote).mkString("[",",","]")
      )
    }
  }

  def +=(ts: DateTime, request: GetOfferRequest, actualizedOffer: ActualizedOffer): Unit = {
    if (actualizedOffer eq null) return
    if (!actualizedOffer.hasActualizationInfo) return
    val current = buffer.get()
    val offer = actualizedOffer.getOffer
    val actualization = actualizedOffer.getActualizationInfo
    val flights = for (flight <- actualization.getFlightsList) yield {
      val fromPoint = flight.getTo.getPointList.head
      val toPoint = flight.getTo.getPointList.last
      val backFromPoint = flight.getTo.getPointList.head
      val backToPoint = flight.getTo.getPointList.last
      Flight(
        flight.getAdditionalPrice,
        fromPoint.getCompany,
        fromPoint.getFlightNumber,
        fromPoint.getGeoId,
        fromPoint.getAirport,
        fromPoint.getAirportCode,
        toPoint.getGeoId,
        toPoint.getAirport,
        toPoint.getAirportCode,
        backFromPoint.getCompany,
        backFromPoint.getFlightNumber,
        backFromPoint.getGeoId,
        backFromPoint.getAirport,
        backFromPoint.getAirportCode,
        backToPoint.getGeoId,
        backToPoint.getAirport,
        backToPoint.getAirportCode
      )
    }

    current += Record(
      ts, request.tourId,
      request.hotelRequest.from,
      offer.getHotelId,
      offer.getDate.toLocalDate,
      offer.getNights,
      request.hotelRequest.agesSerializable.toArray,
      offer.getPansion.getNumber,
      offer.getOriginalRoomCode,
      offer.getPrice,
      actualization.getPrice,
      actualization.getFuelCharge,
      if (actualization.hasVisaPrice) Some(actualization.getVisaPrice) else None,
      if (actualization.hasInfantPrice) Some(actualization.getInfantPrice) else None,
      flights.toArray
    )
    if (current.size >= flushAfter) flush()
  }

  def flush(sync: Boolean = false): Unit = {
    val current = buffer.getAndSet(new ArrayBuffer[Record](bufferSize))
    if (current.isEmpty) return

    val time = System.currentTimeMillis()
    val f = clickHouseClient.update(s"insert into ${clickHouseClient.database}.actualized FORMAT TabSeparated",
      current.view.map(_.toRowString)
    ).andThen {
      case _ => log.info(s"Updated ${current.size} records in ${System.currentTimeMillis() - time} ms.")
    }
    if (sync) {
      Await.result(f, 1.minute)
    }
  }
}
